import json
from unittest.mock import patch

import pytest
from requests import HTTPError
from set_initial_status import STATUS_TRACKING_URL, SetInitialStatus


class TestSetInitialStatus:
    def setup_method(self):
        self.pipelet = SetInitialStatus({})

    def test_send_status_success(self, mock_requests_post, status_data, caplog):
        self.pipelet._send_status(status_data)

        mock_requests_post.assert_called_with(
            STATUS_TRACKING_URL.format(
                cluster="https://example.com", project_id="project_id", token="token"
            ),
            json=status_data,
            timeout=10,
        )

        assert "test-item" in caplog.text

    def test_send_status_failure(self, mock_requests_post, status_data, caplog):
        mock_requests_post.side_effect = HTTPError
        with pytest.raises(HTTPError):
            self.pipelet._send_status(status_data)
        assert "Failed to set initial status for document test-item" in caplog.text

    def test_get_status_data_incomplete_labels(
        self, item, item_keywords, uploader_info, status_data
    ):
        item["keywords"].update({"document_type": item_keywords["document_type"]})
        status_data.pop("company_name")
        status_data.pop("document_date")
        status_data.pop("wfi_document_id")

        with patch(
            "set_initial_status.SetInitialStatus._get_uploader_info",
            return_value=dict_values_to_list(uploader_info),
        ):
            assert status_data == self.pipelet._get_status_data(item)

    def test_get_status_data(self, item, item_keywords, uploader_info, status_data):
        item["keywords"].update(item_keywords)

        with patch(
            "set_initial_status.SetInitialStatus._get_uploader_info",
            return_value=dict_values_to_list(uploader_info),
        ):
            assert status_data == self.pipelet._get_status_data(item)

    def test_get_status_data_wfi_document(
        self, item, item_keywords, uploader_info, status_data
    ):
        item["keywords"].update(item_keywords)
        item["keywords"]["source_type"] = ["WFI Historic"]

        with patch(
            "set_initial_status.SetInitialStatus._get_uploader_info",
            return_value=dict_values_to_list(uploader_info),
        ):
            status_data["source_type"] = "WFI Migration"
            status_data["updater"] = {
                "lan_id": ["Workbench:SystemAdmin"],
                "email": ["Workbench:SystemAdmin"],
                "name": ["Workbench:SystemAdmin"],
                "role_ocbc": ["Workbench:SystemAdmin"],
            }
            assert status_data == self.pipelet._get_status_data(item)

    def test_no_uploader_info(self):
        self.pipelet.redis_client.hget.return_value = None

        with pytest.raises(Exception):
            self.pipelet._get_uploader_info(uploader_info["email"])

    def test_get_uploader_info(self, uploader_info):
        self.pipelet.redis_client.hget.return_value = json.dumps(uploader_info).encode()

        assert self.pipelet._get_uploader_info(
            uploader_info["email"]
        ) == dict_values_to_list(uploader_info)


@pytest.fixture
def item():
    return {
        "id": "test-item",
        "title": "Test Item",
        "keywords": {
            "source_type": ["User Upload"],
            "user_email": ["test@gmail.com"],
        },
    }


@pytest.fixture
def item_keywords():
    return {
        "document_type": ["ANNUAL REPORT"],
        "company_name": ["Company A"],
        "document_date": ["2022-12-12"],
        "wfi_document_id": ["QWRQWRQ-QWRQWRQW-RQWRQWRQWRQWRQWRQ"],
    }


@pytest.fixture
def status_data(item_keywords, uploader_info):
    return {
        **{key: value[0] for key, value in item_keywords.items()},
        "document_id": "test-item",
        "document_name": "Test Item",
        "project_id": "project_id",
        "source_type": "User Upload",
        "updater": {key: [value] for key, value in uploader_info.items()},
        "retry_create": True,
    }


@pytest.fixture
def uploader_info():
    return {
        "ecm_role": "admin",
        "email": "ocbc-test+1@squirro.com",
        "name": "BBCA Maker 1",
        "job_title": "WCM(WCM 80) RM",
        "lan_id": "A10089321",
        "org_unit_id": "RE078",
        "org_unit_type": "Team",
        "rmcode": "myecm1",
        "role": "BBCA Maker",
        "role_ocbc": "iLMS:BBCACSOMaker",
    }


def dict_values_to_list(data):
    return {key: [value] for key, value in data.items()}
