"""Set initial status."""

import json
from typing import TYPE_CHECKING

import requests

from octopus.clients import init_redis_client, init_squirro_client
from octopus.utils import load_config
from squirro.sdk import PipeletV1, require

if TYPE_CHECKING:
    from logging import Logger
    from typing import Any, TypedDict

    class StatusData(
        TypedDict,
        total=False,  # some keys are optional
    ):
        """Status data to be sent to status tracking studio plugin."""

        document_id: str
        document_name: str
        project_id: str
        source_type: str
        retry_create: bool
        document_type: "str | None"
        company_name: "str | None"
        document_date: "str | None"
        updater: "dict[str, list[str]]"


STATUS_TRACKING_URL = (
    "{cluster}/studio/document_status_tracking/projects/"
    "{project_id}/documents?token={token}"
)
USERS_HASH = "user_roles_hash"
RETRY_COUNT = 3


@require("log")
class SetInitialStatus(PipeletV1):  # type: ignore[misc]
    """Set initial status on the document.

    Set the appropriate status(es) for the document based on the source,
    user, and if any of the labels (Document Type, Company Name, and
    Document Date) is missing.
    """

    log: "Logger"

    def __init__(self, _: "dict[str, Any]") -> None:
        """Initialize the pipelet."""
        self.config = load_config()
        self.sq_client, self.project_id = init_squirro_client(self.config)
        self.redis_client = init_redis_client()

    def consume(self, item: "dict[str,Any]") -> "dict[str, Any]":
        """Consume an item.

        Args:
            item: The item to consume

        Returns:
            The consumed item
        """
        status_data = self._get_status_data(item)
        self._send_status(status_data)

        return item

    def _get_status_data(self, item: "dict[str, Any]") -> "StatusData":
        """Get status data for the document.

        Creates and formats a dictionary to be sent to the status tracking
        studio plugin to set the initial status.

        Args:
            item: Current item.

        Returns:
            The status data to be sent to the status tracking studio plugin.
        """
        status_data: StatusData = {
            "retry_create": True,
            "document_id": item["id"],
            "document_name": item["title"],
            "project_id": self.project_id,
        }

        keywords: dict[str, list[str]] = item["keywords"]
        if (source_type := keywords["source_type"][0]).startswith("WFI"):
            status_data["source_type"] = "WFI Migration"
            status_data["updater"] = {
                "lan_id": ["Workbench:SystemAdmin"],
                "email": ["Workbench:SystemAdmin"],
                "name": ["Workbench:SystemAdmin"],
                "role_ocbc": ["Workbench:SystemAdmin"],
            }
        else:
            status_data["source_type"] = source_type
            status_data["updater"] = self._get_uploader_info(keywords["user_email"][0])

        for label in (
            "document_type",
            "company_name",
            "document_date",
            "wfi_document_id",
        ):
            if value := keywords.get(label, []):
                status_data[label] = value[0]  # type: ignore[literal-required]

        return status_data

    def _get_uploader_info(self, email: str) -> "dict[str, list[str]]":
        """Retrieves the user information from redis user_roles_hash.

        Args:
            email: The email address of the uploader.

        Returns:
            The user's data stored in the redis.

        Raises:
            ValueError: The email is not found in redis.
        """
        uploader_info_raw = self.redis_client.hget(USERS_HASH, email.lower())
        if not uploader_info_raw:
            msg = f"Email {email} not found in User API"
            self.log.error(msg)
            raise ValueError(msg)
        uploader_info = json.loads(uploader_info_raw.decode())
        return {key: [value] for key, value in uploader_info.items()}

    def _send_status(self, data: "StatusData") -> None:
        """Sends the status data to the status tracking studio plugin.

        Args:
            data: The data to be sent.

        Raises:
            HTTPError: Error sending the requests after retry attempts.
        """
        item_id = data["document_id"]
        self.log.info("Sending status data for document %s", item_id)

        try:
            res = requests.post(
                STATUS_TRACKING_URL.format(
                    cluster=self.config["squirro"]["cluster"],
                    project_id=self.project_id,
                    token=self.config["squirro"]["token"],
                ),
                json=data,
                timeout=10,
            )
            res.raise_for_status()
            self.log.info("%s - %s", item_id, res.json()["message"])
        except requests.HTTPError:
            self.log.exception("Failed to set initial status for document %s", item_id)
            raise
