import json
from datetime import datetime, timedelta
from unittest.mock import ANY

import pytest
from retry_storing_into_wfi import REDIS_FAILED_ITEMS_HASH, RetryStoringIntoWFI

from octopus.activity_tracking import RejectionReason


class TestRetryStoringIntoWFI:
    def setup_method(self):
        self.pipelet = RetryStoringIntoWFI({})
        self.content_url = "storage://localfile/12/34/56/PDF_609.pdf"

    def test_invalid_redis_item(self, mock_redis_client):
        assert self.pipelet.consume({"id": "item-id"}) is None  # No title

        mock_redis_client.hget.return_value = None
        assert self.pipelet.consume({"title": self.content_url}) is None

    def test_retry_window_exceeded(self, mock_redis_client, redis_failed_item, caplog):
        old_date = (datetime.now() - timedelta(days=3)).strftime("%Y-%m-%dT%H:%M:%S")
        redis_failed_item["item"]["created_at"] = old_date
        mock_redis_client.hget.return_value = json.dumps(redis_failed_item).encode()

        output = self.pipelet.consume({"title": self.content_url})

        redis_failed_item["item"]["keywords"]["wfi_status"] = ["Failure"]
        assert output == redis_failed_item["item"]
        assert "Item `test-item` was created more than 24 hours ago" in caplog.text
        mock_redis_client.hset.assert_called_once_with(
            self.pipelet.item_rejection_hash,
            "test-item",
            RejectionReason.CREATED_OVER_24H,
        )
        mock_redis_client.hdel.assert_called_once_with(
            REDIS_FAILED_ITEMS_HASH, self.content_url
        )

    def test_retry_success(self, mock_redis_client, redis_failed_item):
        mock_redis_client.hget.return_value = json.dumps(redis_failed_item).encode()

        output = self.pipelet.consume({"title": self.content_url})

        mock_redis_client.hdel.assert_any_call(
            REDIS_FAILED_ITEMS_HASH, self.content_url
        )
        mock_redis_client.hdel.assert_any_call(
            self.pipelet.item_rejection_hash, "test-item"
        )
        mock_redis_client.hset.assert_called_once_with(
            self.pipelet.item_success_hash, "test-item", ANY
        )
        assert output["keywords"]["wfi_status"] == ["Success"]


@pytest.fixture
def redis_failed_item():
    return {
        "item": {
            "files": [
                {
                    "name": "PDF_609.pdf",
                    "mime_type": "application/pdf",
                    "id": "file-pdf-0",
                    "content_url": "storage://localfile/12/34/56/PDF_609.pdf",
                },
                {
                    "id": "file-img-0",
                    "mime_type": "image/png",
                    "content_url": "storage://localfile/12/34/56/PDF_609.pdf.preview.png",
                },
            ],
            "title": "PDF_609.pdf",
            "keywords": {
                "source_type": ["User Upload"],
                "force_ocr": ["true"],
                "document_type_pred": ["Resolution"],
                "document_date_pred": ["2024-05-01"],
                "document_type": ["Resolution"],
                "document_date": ["2024-05-01"],
                "wfi_company_cif": [""],
                "wfi_company_name": [""],
                "wfi_company_segment": ["unassigned"],
                "wfi_company_rm_code": ["NA"],
                "wfi_company_team_name": ["NA"],
                "wfi_document_category": ["Unclassified"],
                "wfi_document_name": ["Unclassified"],
                "wfi_document_type": ["Unclassified"],
                "wfi_document_date": ["2024-05-01"],
                "wfi_references": [""],
            },
            "created_at": datetime.now().strftime("%Y-%m-%dT%H:%M:%S"),
            "id": "test-item",
        },
        "original_file": {
            "name": "PDF_609.pdf",
            "mime_type": "application/pdf",
            "id": "file-pdf-0",
            "content_url": "storage://localfile/12/34/56/PDF_609.pdf",
        },
        "wfi_document_id": "",
    }
