"""Map WFI labels to Squirro labels."""

from datetime import datetime
from typing import TYPE_CHECKING

import pytz

from octopus.data.company_data import CompanyDataIndex
from squirro.sdk import PipeletV1, require

if TYPE_CHECKING:
    from logging import Logger
    from typing import Any

# flake8: noqa: E501
# pylint: disable=line-too-long
_MAPPINGS = {
    "Accession Agreement": "OTHER AGREEMENT",
    "Acknowledgement for Production of Prior Titles": "MORTGAGE",
    "Acknowledgement Notice of Assignments (Equipment/Vehicles)": "ASSIGNMENT",
    "Acknowledgement Of Notice of Assignments": "ASSIGNMENT",
    "Acknowledgement of Notice of Assignments (RE)": "ASSIGNMENT",
    "Agreement for Lease": "TENANCY",
    "Agreement of LC discounting facility and NDF Transactions": "OTHER AGREEMENT",
    "Aircraft Mortgage": "MORTGAGE",
    "Appointment of Process Agent": "CONFIRMATION/CONSENT",
    "Approval Memos / Emails": "APPROVALS",
    "Assignment of Account Receivables": "ASSIGNMENT",
    "Assignment of Building Agreement": "ASSIGNMENT",
    "Assignment of Building Contract": "ASSIGNMENT",
    "Assignment of Debtors": "ASSIGNMENT",
    "Assignment of Earnings": "ASSIGNMENT",
    "Assignment of Insurances": "ASSIGNMENT",
    "Assignment of Insurances (RE)": "ASSIGNMENT",
    "Assignment of Interest Escrow Account": "ASSIGNMENT",
    "Assignment of Performance Bonds": "ASSIGNMENT",
    "Assignment of Proceeds": "ASSIGNMENT",
    "Assignment of Project Documents": "ASSIGNMENT",
    "Assignment of Refund Guarantee": "ASSIGNMENT",
    "Assignment of Rental and Sale Proceeds": "ASSIGNMENT",
    "Assignment of Rental Proceeds": "ASSIGNMENT",
    "Assignment of Shipbuilding Contract": "ASSIGNMENT",
    "Bankers Guarantee/ SBLC": "BANK GUARANTEE/SBLC",
    "Banker's Guarantee/ SBLC": "BANK GUARANTEE/SBLC",
    "Bill of Lading": "EXTERNAL CORRESPONDENCES",
    "Building Agreement": "MORTGAGE",
    "Cash and Management Agreement": "OTHER AGREEMENT",
    "Caveat": "MORTGAGE",
    "Certificate of Incorporation and Memorandum & Articles of Association": "INCORPORATION DOC",
    "Certificate of Title/ Subsidiary Strata Certificate of Title": "TITLE DEED",
    "Charge of Cash & Security Agreement": "CHARGE",
    "Charge over Accounts": "CHARGE",
    "Charter Contract": "CONTRACT",
    "Confirmation of Charge from CDP": "REGISTRATION OF SECURITY",
    "Credit Memo": "CM",
    "Credit Proposition": "CP",
    "Credit Review": "CR",
    "Debenture": "DEBENTURE",
    "Deed of Assignment/ Master Agreement": "ASSIGNMENT",
    "Deed of Charge": "CHARGE",
    "Deed of Conveyance/Reconveyance": "MORTGAGE",
    "Deed of Covenants": "DEED OF COVENANTS",
    "Deed of Guarantee and Indemnity (Limited Liability)": "CORPORATE GUARANTEE",
    "Deed of Guarantee and Indemnity (Unlimited Liabili": "CORPORATE GUARANTEE",
    "Deed of Guarantee and Indemnity (Unlimited Liability)": "CORPORATE GUARANTEE",
    "Deed of Subordination": "SUBORDINATION DEED",
    "Deed of Subordination (RE)": "SUBORDINATION DEED",
    "Deed of Undertaking and Subordination": "SUBORDINATION DEED",
    "Deed/Letter of Subordination": "SUBORDINATION DEED",
    "Deed/Letter of Undertaking": "UNDERTAKING",
    "Delivery Order": "EXTERNAL CORRESPONDENCES",
    "Direct Debit Authorization Form": "LETTER - OTHERS",
    "Directors Resolution": "RESOLUTION",
    b"Directors\xe2\x80\x99 Resolution".decode("utf8"): "RESOLUTION",
    "Discharge Documents (Partial/ Total)": "DISCHARGE",
    "Email correspondences within the Bank": "APPROVAL",
    "External correspondences eg. Letters from Customer": "EXTERNAL CORRESPONDENCES",
    "External correspondences eg. Letters from Customers": "EXTERNAL CORRESPONDENCES",
    "Facility Agreement": "FACILITY AGREEMENT",
    "Factoring Agreement": "OTHER AGREEMENT",
    "Fiduciary Agreement": "OTHER AGREEMENT",
    "Financial Reports": "ANNUAL REPORT",
    "Fixed and Floating Charge": "CHARGE",
    "Fixed Charge": "CHARGE",
    "Floating Charge": "CHARGE",
    "Foreign Language Documents": "LETTER - OTHERS",
    "Form 5": "INTERNAL DOCUMENTS (BBCA)",
    "Form 6 - CR Checklist": "INTERNAL DOCUMENTS (BBCA)",
    "Form of Confirmation and Consent": "CONFIRMATION/CONSENT",
    "General Assignment": "ASSIGNMENT",
    "General Assignment (Misc)": "ASSIGNMENT",
    "Grant of Probate": "POWER OF ATTORNEY",
    "Hire Purchase Agreement": "OTHER AGREEMENT",
    "Indenture of Mortgage": "MORTGAGE",
    "Insurance": "INSURANCE",
    "Intecreditors Agreement": "INTERCREDITOR AGREEMENT",
    "ISDA": "ISDA MASTER AGREEMENT/SCHEDULE",
    "Landlord Waiver": "LETTER - OTHERS",
    "Lease": "MORTGAGE",
    "Legal Opinion": "LEGAL OPINION",
    "Letter of Advice": "LETTER OF OFFER",
    "Letter of Authority and Indemnity in respect of telefaxed instructions": "LETTER - OTHERS",
    "Letter of Comfort and Awareness": "UNDERTAKING",
    "Letter of Offer": "LETTER OF OFFER",
    "Letter of Set-Off /Letter of Set-off and Charge": "CHARGE",
    "Letter of Undertaking": "UNDERTAKING",
    "Letter/ email with lawyer": "LEGAL CORRESPONDENCES",
    "LMA Agreement": "OTHER AGREEMENT",
    "Lodgment Memorandum": "INTERNAL DOCUMENTS (BBCA)",
    "Master Assignment": "ASSIGNMENT",
    "Master Trust Receipt Agreement (MTRA)": "OTHER AGREEMENT",
    "Memo/ Loading Forms for line implementation": "INTERNAL DOCUMENTS (BBCA)",
    "Memorandum of Charge / Assignment": "CHARGE",
    "Mortgage in Escrow": "MORTGAGE",
    "Mortgage Reducing Term Assignment (MRTA)": "FACILITY AGREEMENT",
    "Motor Inspection reports": "LETTER - OTHERS",
    "Negative Pledge Letter/ Agreement": "CONFIRMATION/CONSENT",
    "Notice of Assignment": "ASSIGNMENT",
    "Notice of Assignment (Equipment/Vehicles)": "ASSIGNMENT",
    "Notice of Assignment (RE)": "ASSIGNMENT",
    "Novation Agreement": "OTHER AGREEMENT",
    "Open or Closed Mortgage": "MORTGAGE",
    "Option to Purchase": "CONTRACT",
    "Original Tax Invoice": "EXTERNAL CORRESPONDENCES",
    b"Other banks\xe2\x80\x99 account statements".decode("utf8"): "LETTER - OTHERS",
    "Pledge Agreement": "PLEDGE",
    "Power of Attorney": "POWER OF ATTORNEY",
    "Proportionate Guarantee": "CORPORATE GUARANTEE",
    "Purchase Contract": "CONTRACT",
    "Redemption checklist/ instruction": "INTERNAL DOCUMENTS (BBCA)",
    "Refund Gte/ Performance Gte": "BANK GUARANTEE/SBLC",
    "Report On Title": "LEGAL CORRESPONDENCES",
    "Risk Disclosure Statement for Treasury and Financial Derivatives": "ISDA MASTER AGREEMENT/SCHEDULE",
    "Sale & Purchase Agreement": "CONTRACT",
    "Searches": "SEARCHES",
    "Security Sharing Deed": "SECURITY TRUST/DEED",
    "Security Sharing Deed (RE)": "SECURITY TRUST/DEED",
    "Security Trust Deed (Misc)": "SECURITY TRUST/DEED",
    "Security Trust Deed (RE)": "SECURITY TRUST/DEED",
    "Share Certificate": "SHARE CERTIFICATE/TRANSFER FORMS",
    "Share Charge": "CHARGE",
    "Share Transfer Form": "SHARE CERTIFICATE/TRANSFER FORMS",
    "Shareholders Resolution": "RESOLUTION",
    "Shareholders' Resolution": "RESOLUTION",
    "Shares Miscellaneous": "SHARE CERTIFICATE/TRANSFER FORMS",
    "Ship Management Contract/ Agreement": "CONTRACT",
    "Shipbuilding Contract/ Agreement": "CONTRACT",
    "Singapore Land Authority Letter": "LETTER - OTHERS",
    "Specific Charge": "CHARGE",
    "Spousal Consent": "CONFIRMATION/CONSENT",
    "Statement Containing Particulars of Charge with ACRA's email confirmation of charge (including form 33,34 &40) (Debenture/ Charge)": "REGISTRATION OF SECURITY",
    "Statement Containing Particulars of Charge with ACRA's email confirmation of charge (including form 33,34 &40) (Equipment/Vehicles)": "REGISTRATION OF SECURITY",
    "Statement Containing Particulars of Charge with ACRA's email confirmation of charge (including form 33,34 &40) (Shares)": "REGISTRATION OF SECURITY",
    "Statement Containing Particulars of Charge with ACRA's email confirmation of charge (including form 33,34 &40) (Vessels/Aircrafts/Assignment)": "REGISTRATION OF SECURITY",
    "Statement Containing Particulars of Charge with ACRA's email confirmation of charge (including Forms 33, 34 & 40) (Mortgage/Assignment)": "REGISTRATION OF SECURITY",
    "Statement Containing Particulars of Charge with ACRAs email confirmation of charge (including form 33,34 &40) (Debenture/ Charge)": "REGISTRATION OF SECURITY",
    "Statement Containing Particulars of Charge with ACRAs email confirmation of charge (including form 33,34 &40) (Shares)": "REGISTRATION OF SECURITY",
    "Statement Containing Particulars of Charge with ACRAs email confirmation of charge (including form 33,34 &40) (Vessels/Aircrafts/Assignment)": "REGISTRATION OF SECURITY",
    "Statement Containing Particulars of Charge with ACRAs email confirmation of charge (including Forms 33, 34 & 40) (Mortgage/Assignment)": "REGISTRATION OF SECURITY",
    "Supplemental Agreement": "OTHER AGREEMENT",
    "Supplemental Deed": "OTHER AGREEMENT",
    "Syndication Agreement": "SYNDICATION AGREEMENT",
    "Tenancy Agreement": "TENANCY",
    "Term Loan Agreement": "FACILITY AGREEMENT",
    "Transcript of Vessel (including trading certificates)": "TRANSCRIPT OF VESSEL",
    "Transfer Certificate": "OTHER AGREEMENT",
    "Undertaking Deed": "UNDERTAKING",
    "Valuation Report": "VALUATION",
    "Variation of Mortgage": "MORTGAGE",
    "Vehicle Registration Evidence from LTA": "REGISTRATION OF SECURITY",
    "Vessel Mortgage": "MORTGAGE",
    "Withdrawal Memorandum": "INTERNAL DOCUMENTS (BBCA)",
}


# pylint: disable=too-few-public-methods
@require("log")
class MapWFILabelsToSquirro(PipeletV1):  # type: ignore[misc]
    """Map WFI labels to Squirro labels.

    Map WFI labels to Squirro labels.
    """

    log: "Logger"

    def __init__(self, _: "dict[str, Any]") -> None:
        """Initialize the pipelet."""
        self.company_index = CompanyDataIndex.load_index()

    def consume(self, item: "dict[str, Any]") -> "dict[str, Any]":
        """Consume an item.

        Args:
            item: The item to consume

        Returns:
            The consumed item
        """
        item["keywords"].update(self._tag_document_type(item))
        item["keywords"].update(self._tag_company_name(item))
        item["keywords"].update(self._tag_document_date(item))

        return item

    def _tag_company_name(self, item: "dict[str, Any]") -> "dict[str, Any]":
        """Tag the company name.

        Args:
            item: The item to tag

        Returns:
            The tagged item
        """
        if not (wfi_company_cif := item["keywords"].get("wfi_company_cif", [""])[0]):
            self.log.warning(
                "Item `%s` does not contain valid wfi_company_cif.",
                item["id"],
            )
            return {}

        company_data = self.company_index.search_by_cifs(wfi_company_cif)
        if not company_data:
            self.log.warning(
                "Company with cif `%s` for item `%s` does not exist.",
                wfi_company_cif,
                item["id"],
            )
            return {}

        if not (
            company_names := [
                name
                for company in company_data
                if (name := company.get("company_name", [""])[0])
            ]
        ):
            self.log.warning(
                "Company with cif `%s` for item `%s` does not have a company name.",
                wfi_company_cif,
                item["id"],
            )
            return {}

        return {"company_name_true": company_names}

    def _tag_document_date(self, item: "dict[str, Any]") -> "dict[str, Any]":
        """Tag the document date.

        Args:
            item: The item to tag

        Returns:
            The tagged item
        """
        if not (
            wfi_document_date := item["keywords"].get("wfi_document_date", [""])[0]
        ):
            self.log.warning(
                "Item `%s` does not contain wfi_document_date.", item["id"]
            )
            return {}

        try:
            wfi_document_date_formatted = (
                datetime.strptime(
                    wfi_document_date,
                    "%d-%b-%Y %H:%M:%S",  # Date format in WFI
                )
                .astimezone(pytz.timezone("Asia/Singapore"))
                .strftime("%Y-%m-%d")
            )
        except Exception:
            self.log.exception(
                "Failed to parse document date `%s` for item `%s`.",
                wfi_document_date,
                item["id"],
            )
            return {}

        return {"document_date_true": [wfi_document_date_formatted]}

    def _tag_document_type(self, item: "dict[str, Any]") -> "dict[str, Any]":
        """Tag the document type.

        Args:
            item: The item to tag

        Returns:
            The tagged item
        """
        if (
            wfi_document_name := item["keywords"].get("wfi_document_name", [""])[0]
        ) not in _MAPPINGS:
            self.log.warning(
                "Item `%s` does not contain valid wfi_document_name.",
                item["id"],
            )
            return {}

        return {"document_type_true": [_MAPPINGS[wfi_document_name]]}
