# ruff: noqa
# mypy: ignore-errors
import json
import time
from datetime import datetime

import requests

from octopus.clients.redis_client import init_redis_client
from octopus.utils import load_config
from squirro.sdk import PipeletV1, require

MAPPINGS = {
    "DateCreated": "created_at",
    "BBCA_CIF": "wfi_company_cif",
    "BBCA_CustomerName": "wfi_company_name",
    "BBCA_Segment": "wfi_company_segment",
    "BBCA_BusinessUnit": "wfi_company_rm_code",
    "BBCA_Team": "wfi_company_team_name",
    "BBCA_DocumentCategory": "wfi_document_category",
    "BBCA_DocumentType": "wfi_document_type",
    "BBCA_DocumentName": "wfi_document_name",
    "BBCA_DocumentDate": "wfi_document_date",
    "BBCA_Reference": "references",
}


@require("log")
class FetchWFIMetadata(PipeletV1):  # type: ignore[misc]
    """Fetches metadata from WFI.

    Fetches metadata from WFI and create corresponding labels.
    """

    def __init__(self, _) -> None:
        cfg = load_config()

        try:
            self.log.info("Loading WFI config")
            wfi_api_header_name = (
                "wfiapi-octopus-api-key"
                if cfg.get("squirro", "environment") in ("DR", "PROD")
                else "wfiapi-ecoffice-api-key"
            )

            if not (url := cfg.get("wfi", "url")):
                raise ValueError("Missing WFI url in config file")

            if not (port := cfg.get("wfi", "port")):
                raise ValueError("Missing WFI port in config file")

            self.wfi_url = f"{url}:{int(port)}/wfiapi"
            self.headers = {
                "domain": cfg.get("wfi", "domain"),
                "channel": cfg.get("wfi", "channel"),
                "objectstore": cfg.get("wfi", "objectstore"),
                wfi_api_header_name: cfg.get("wfi", "api_key"),
            }
        except Exception:
            self.log.exception("Failed to load WFI api config.")
            raise

        self.redis_client = init_redis_client()

    def consume(self, item: "dict") -> "dict | None":  # type: ignore[type-arg]
        if not (doc_id := item.get("keywords", {}).get("wfi_document_id", [""])[0]):
            msg = "Item has no document_id."
            self.log.exception(msg)
            raise ValueError(msg)

        metadata = self._fetch_metadata(doc_id)
        created_at = metadata.pop("created_at", "")
        if created_at:
            item["created_at"] = datetime.strptime(
                created_at, "%d-%b-%Y %H:%M:%S"
            ).isoformat()
        item["keywords"].update(metadata)

        return item

    def _add_to_redis(self, doc_id: str) -> None:
        self.redis_client.zadd("failed_wfi_to_squirro", {doc_id: int(time.time())})

    def _fetch_metadata(self, doc_id: str) -> "dict[str, str]":
        """Get document metadata from WFI.

        Args:
            doc_id (str): Document id.

        Returns:
            Dict: Document metadata.

        Raises:
            Exception: Error fetching document from WFI.
        """
        try:
            res: requests.models.Response = requests.post(
                f"{self.wfi_url}/fetchmetadatabydocid/{{{doc_id}}}",
                json=list(MAPPINGS.keys()),
                headers=self.headers,
                timeout=10,
            )
            res.raise_for_status()
        except Exception:
            self.log.exception(f"Error fetching document {doc_id} from WFI.")
            self._add_to_redis(doc_id)
            raise

        try:
            content = json.loads(res.content.decode())
            if "propertyNameValuePair" not in content:
                msg = "Response does not contain metadata."
                self.log.exception(msg)
                raise ValueError(msg)
            metadata = content["propertyNameValuePair"]
            del metadata["Id"]  # Already have the id.
        except Exception:
            self.log.exception("Error extracting metadata from response.")
            self._add_to_redis(doc_id)
            raise

        output = {}
        for wfi_key, sq_key in MAPPINGS.items():
            if wfi_key in metadata:
                if wfi_key == "BBCA_Reference" and not metadata[wfi_key]:
                    continue
                output[sq_key] = metadata[wfi_key]

        return output
