# pylint: skip-file
# mypy: ignore-errors
import json
from datetime import datetime

from octopus.clients.redis_client import init_redis_client
from squirro.sdk import PipeletV1, require

REDIS_ITEM_HASH = "item_hash"

UNKNOWN = "UNKNOWN"


@require("log")
class ActivityTrackingIDs(PipeletV1):
    """Store item metadata needed for Activity Tracking and email
    notifications.
    """

    def __init__(self, config: "dict") -> None:
        self.redis_client = init_redis_client()

    def consume(self, item: "dict") -> "dict":
        """Consume an item.

        Args:
            item: The item to consume

        Returns:
            The consumed item
        """
        # User info available in email dataloader, widget and WFI
        global REDIS_ITEM_HASH

        item_id = item.get("id", "")
        keywords = item.get("keywords", {})
        source_type = keywords.get("source_type", keywords.get("method", [UNKNOWN]))[0]

        try:
            created_at = item["created_at"]
        except KeyError:
            created_at = datetime.utcnow().replace(microsecond=0).isoformat()

        if source_type in ("User Upload", "Email"):
            user_name = keywords.get("user_name", [UNKNOWN])[0]
            user_email = keywords.get("user_email", [UNKNOWN])[0]
        else:
            user_name = UNKNOWN
            user_email = UNKNOWN

        metadata = {
            "source_type": source_type,
            "title": item.get("title", "No title"),
            "created_at": created_at,
            "user_name": user_name,
            "user_email": user_email,
        }

        if source_type.startswith("WFI"):
            REDIS_ITEM_HASH = "item_wfi_hash"
            wfi_id = item["keywords"].get("wfi_id_original", [""])[0]
            metadata["wfi_id_original"] = wfi_id

        message = f"Adding item `{item_id}` to Redis hash `{REDIS_ITEM_HASH}`"

        try:
            self.redis_client.hsetnx(REDIS_ITEM_HASH, item_id, json.dumps(metadata))
        except Exception:
            self.log.exception("%s failed", message)

        self.log.info(message)

        return item
