import json
from unittest import mock

import pytest
from vault import EmptyValueError, InvalidValueError, _get_input, main

from octopus.utils import load_vault


@mock.patch("vault.getpass", return_value="password")
def test_get_input(_):
    _get_input()

    assert _get_input() == "password"


@mock.patch("vault.getpass", return_value="")
def test_get_input_empty(_):
    with pytest.raises(EmptyValueError):
        _get_input()


@mock.patch("vault.getpass", side_effect=["password", "pass"])
def test_get_input_mismatch(_):
    with pytest.raises(InvalidValueError):
        _get_input()


@mock.patch("vault.input", return_value="n")
@mock.patch("vault.getpass", return_value="password")
def test_main_not_updated(_, __, tmp_path, mock_load_vault):
    vault_path = tmp_path / "vault.json"
    with vault_path.open("w+", encoding="utf-8") as fp:
        json.dump(mock_load_vault.return_value, fp, indent=2)
    main("section", "field", vault_path=str(vault_path))

    assert mock_load_vault.return_value == load_vault(str(vault_path))


@mock.patch("vault.getpass", return_value="password")
def test_main_new_field(_, tmp_path):
    vault_path = tmp_path / "vault.json"
    vault_path.touch()
    main("section", "new", vault_path=str(vault_path))

    vault = load_vault(str(vault_path))
    assert "new" in vault["section"]
    assert vault["section"]["new"]


@mock.patch("vault.input", return_value="y")
@mock.patch("vault.getpass", return_value="password")
def test_main_existing_field(_, __, tmp_path, mock_load_vault):
    vault_path = tmp_path / "vault.json"
    vault_path.touch()
    old_val = mock_load_vault.return_value["section"]["field"]
    main("section", "field", vault_path=str(vault_path))

    vault = load_vault(str(vault_path))
    assert old_val != vault["section"]["field"]
