"""Vault tool to update encrypted values in the vault."""

import argparse
import json
import logging
from getpass import getpass
from pathlib import Path

from octopus.utils import encrypt, load_key, load_vault, set_log_verbosity

set_log_verbosity(logging.INFO)


class EmptyValueError(Exception):
    """Exception raised for empty value."""

    def __init__(self) -> None:
        """Initialize the exception."""
        super().__init__("Value cannot be empty.")


class InvalidValueError(Exception):
    """Exception raised for invalid value."""

    def __init__(self) -> None:
        """Initialize the exception."""
        super().__init__("Value is not valid.")


def _get_input() -> str:
    """Get user input.

    Returns:
        The value entered by the user

    Raises:
        EmptyValueError: If the value is empty
        InvalidValueError: If the value is not valid
    """
    val = getpass("Enter value: ")
    if not val:
        raise EmptyValueError

    if val != getpass("Re-enter value: "):
        raise InvalidValueError

    return val


def main(
    section: str, field: str, vault_path: str = "/etc/squirro/.vault.json"
) -> None:
    """Entrypoint."""
    vault = load_vault()
    key = load_key()
    vault_fp = Path(vault_path)

    val = _get_input()

    updated = False
    if section not in vault:
        encrypted = encrypt(val, key)
        vault[section] = {field: encrypted}
        updated = True
    elif field not in vault[section] or (
        input("Field already exists. Update? [y/n]: ").lower() == "y"
    ):
        encrypted = encrypt(val, key)
        vault[section][field] = encrypted
        updated = True

    if updated:
        with vault_fp.open("w", encoding="utf-8") as fp:
            json.dump(vault, fp, indent=2)
        logging.warning("Vault `%s` updated.", vault_fp)
        print(  # noqa: T201
            "Please put the following line to as the value on the relevant file where"
            "the value would be read:\n"
            f"CK_CMD::jsonfile:{vault_fp.as_posix()}:{section}:{field}"
        )


if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument(
        "--section", type=str, required=True, help="Section to be updated."
    )
    parser.add_argument("--field", type=str, required=True, help="Field to be updated.")
    args = parser.parse_args()
    main(args.section, args.field)
