"""Initialize vault."""

import json
import logging
from collections import defaultdict
from configparser import ConfigParser

from cryptography import fernet

from octopus.utils import check_file_exists, encrypt, load_config, set_log_verbosity

set_log_verbosity(logging.INFO)


VALUES_TO_ENCRYPT = {
    "../../config/main.ini": [
        ("squirro", "token"),
        ("wfi", "api_key"),
    ],
    "/etc/squirro/common.ini": [
        ("dataloader", "exchange_client_id"),
        ("dataloader", "exchange_client_secret"),
        ("dataloader", "exchange_tenant_id"),
        ("index", "es_auth_password"),
        ("redis", "password"),
        ("redis_cache", "password"),
    ],
    "/etc/squirro/configuration.ini": [("configuration", "db")],
    "/etc/squirro/datasource.ini": [
        ("datasource", "db"),
    ],
    "/etc/squirro/emailsender.ini": [
        ("emailsender", "db"),
    ],
    "/etc/squirro/filtering.ini": [
        ("filtering", "db"),
    ],
    "/etc/squirro/fingerlogging.info.ini": [("fingerlogging.info", "db")],
    "/etc/squirro/fingerprint.ini": [("fingerprint", "db")],
    "/etc/squirro/frontend.ini": [
        ("flask_app", "secret_key"),
        ("studio", "db"),
    ],
    "/etc/squirro/machinelearning.ini": [
        ("machinelearning", "db"),
    ],
    "/etc/squirro/notes.ini": [("notes", "db")],
    "/etc/squirro/plumber.ini": [("plumber", "db")],
    "/etc/squirro/scheduler.ini": [("scheduler", "db")],
    "/etc/squirro/topic.ini": [("topic", "db")],
    "/etc/squirro/trends.ini": [("trends", "db")],
    "/etc/squirro/user.ini": [("user", "db")],
}

UPDATE_ENCRYPTED_VALUES: "dict[str, list[tuple[str, str, str, str]]]" = {
    "/etc/squirro/userproxy.ini": [
        ("userproxy", "db", "/etc/squirro/user.ini", "user")
    ],
    "/etc/squirro/topicproxy.ini": [
        ("topicproxy", "db", "/etc/squirro/topic.ini", "topic"),
    ],
}


def load_key(key_path: str) -> str:
    """Load key from file if exists, else create a new key.

    Args:
        key_path: Path to key file.

    Returns:
        Key
    """
    key_fp, exist = check_file_exists(key_path)
    if exist:
        key = key_fp.read_text("utf-8")
        logging.info("Using existing key from %s", key_fp)
    else:
        key_fp.parent.mkdir(parents=True, exist_ok=True)
        key = fernet.Fernet.generate_key().decode()
        key_fp.write_text(key, "utf-8")
        logging.info("Writing key to %s", key_fp)
    return key


def get_db_ip(db: str) -> "tuple[str, str]":
    """Get IP address from DB connection string.

    Args:
        db: DB connection string.

    Returns:
        IP address and password.
    """
    ip = db.split("@")[-1].split("/")[0]
    password = "@".join(db.split("@")[:-1]).split(":")[-1]
    return ip, password


def _update_reused_encrypted_values() -> None:
    """Update encrypted values for services that reuse the same value."""
    for filepath, values in UPDATE_ENCRYPTED_VALUES.items():
        path, exist = check_file_exists(filepath)
        if not exist:
            continue
        logging.info("Reading %s", path)
        cfg = ConfigParser()
        cfg.read(path)
        for section, service, cpath, csection in values:
            c = ConfigParser()
            c.read(cpath)
            cfg[section][service] = c[csection][service]
        logging.info("Writing %s", path)
        with path.open("w", encoding="utf-8") as fp:
            cfg.write(fp)


def main() -> None:
    """Entrypoint."""
    key = load_key("/etc/squirro/.squirro.key")
    vault_path = "/etc/squirro/.vault.json"
    vault_fp, exist = check_file_exists(vault_path)
    vault_fp.parent.mkdir(parents=True, exist_ok=True)
    vault: dict[str, dict[str, str]] = (
        json.loads(vault_fp.read_text("utf-8")) if exist else defaultdict(dict)
    )
    env = load_config()["squirro"]["environment"]

    for filepath, items in VALUES_TO_ENCRYPT.items():
        path, exist = check_file_exists(filepath)
        if not exist:
            continue
        logging.info("Reading %s", path)

        cfg = ConfigParser()
        cfg.read(path)

        # Section corresponds to the section in the .ini files. For example,
        # [wfi] is the section and url will be the service under the section
        # to be updated
        for section, service in items:
            if (
                not cfg.has_section(section)
                or service not in cfg[section]
                or not cfg[section][service]
                or cfg[section][service].startswith("CK_CMD::")
            ):
                continue

            if section not in vault:
                vault[section] = {}

            # Only encrypt if not already encrypted
            if service not in vault[section]:
                val = cfg[section][service]
                if service == "db":
                    if "db_ip" not in vault[section]:
                        vault["db_ip"] = {}
                    ip, password = get_db_ip(val)
                    val = password
                    if env not in vault["db_ip"]:
                        vault["db_ip"][env] = ip
                vault[section][service] = encrypt(val, key)

            cfg[section][service] = f"CK_CMD::jsonfile:{vault_path}:{section}:{service}"
            logging.info("CK_CMD::jsonfile:%s:%s:%s", vault_path, section, service)
        logging.info("Writing %s", path)
        with path.open("w", encoding="utf-8") as fp:
            cfg.write(fp)

    # Save vault as json
    logging.info("Writing vault file to %s", vault_path)
    vault_fp.write_text(json.dumps(vault, indent=2), "utf-8")

    _update_reused_encrypted_values()


if __name__ == "__main__":
    main()
