"""Fetch values from the vault."""

import argparse
import json
import logging
import sys
import urllib.parse
from pathlib import Path
from typing import TYPE_CHECKING

from octopus.utils import decrypt, load_config, load_key, load_vault, set_log_verbosity

if TYPE_CHECKING:
    from typing import Any


set_log_verbosity(logging.INFO)


def _load_json(file_path: "Path") -> "dict[str, Any]":
    """Load JSON file.

    Returns:
        JSON file content.
    """
    with file_path.open(encoding="utf-8") as fp:
        data: dict[str, Any] = json.load(fp)
    return data


def _decrypt_vault(
    key_path: str, vault_path: Path, config: list[str], env: str
) -> None:
    """Decrypt vault.

    Args:
        key_path: The path to the key file
        vault_path: The path to the vault file
        config: The config
        env: The environment

    Raises:
        ValueError: If the config does not have section and key
        KeyError: If the key does not exist in the vault
    """
    logging.info("Reading vault file %s", vault_path)
    vault = load_vault(vault_path.as_posix())
    key = load_key(key_path)

    if len(config) != 2:  # noqa: PLR2004
        msg = f"Config `{config}` needs to have section and key."
        logging.error(msg)
        raise ValueError(msg)

    if not (value := vault.get(config[0].strip())):
        msg = f"Key {config[0]} does not exist in the vault."
        logging.error(msg)
        raise KeyError(msg)

    if not (subvalue := value.get(config[1].strip())):
        msg = f"Key {config[1]} does not exist in section {config[0]}."
        logging.error(msg)
        raise KeyError(msg)

    out = decrypt(subvalue, key)
    if config[1] == "db":
        out = urllib.parse.quote(out)
        con = f"{config[0]}:{out}@{vault['db_ip'][env]}/{config[0]}?"
        if env == "INTERNAL":
            out = f"mysql+mariadbconnector://{con}charset=utf8"
        else:
            out = f"postgresql://{con}"
    sys.stdout.write(out)


def main(key_path: str) -> None:
    """Entrypoint.

    Args:
        key_path: The path to the key file

    Raises:
        FileNotFoundError: If the file does not exist
        NotImplementedError: If the file type is not supported
    """
    (file_type, path, *config) = sys.stdin.read().split(":")
    logging.info("Reading %s file %s", file_type, path)
    logging.info("Config: %s", config)
    env = load_config()["squirro"]["environment"]

    file_path = Path(path)
    if not file_path.exists():
        msg = f"File `{path}` does not exist."
        logging.error(msg)
        raise FileNotFoundError(msg)

    if file_type == "jsonfile":
        if "vault" in file_path.name:
            _decrypt_vault(key_path, file_path, config, env)
        else:
            logging.info("Reading json file %s", file_path)
            data = _load_json(file_path)
            sys.stdout.write(data[config[0]])
    else:
        msg = f"File type `{file_type}` is not supported."
        logging.error(msg)
        raise NotImplementedError(msg)


if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument(
        "--key",
        default="/etc/squirro/.squirro.key",
        type=str,
        help="Path to store key file.",
    )
    args = parser.parse_args()
    main(args.key)
