"""Deinitialize vault."""

import json
import logging
from collections import defaultdict
from configparser import ConfigParser

from cryptography import fernet

from octopus.utils import check_file_exists, decrypt, set_log_verbosity

set_log_verbosity(logging.INFO)


VALUES_TO_ENCRYPT = {
    "../../config/main.ini": [
        ("squirro", "token"),
        ("wfi", "api_key"),
        ("wfi", "url"),
    ],
    "/etc/squirro/common.ini": [
        ("dataloader", "exchange_client_id"),
        ("dataloader", "exchange_client_secret"),
        ("dataloader", "exchange_tenant_id"),
        ("index", "es_auth_user"),
        ("index", "es_auth_password"),
        ("redis", "password"),
        ("redis_cache", "password"),
        ("proxy", "proxy"),
        ("proxy", "no_proxy"),
    ],
    "/etc/squirro/configuration.ini": [("configuration", "db")],
    "/etc/squirro/datasource.ini": [
        ("datasource", "db"),
    ],
    "/etc/squirro/emailsender.ini": [
        ("emailsender", "db"),
        ("emailsender", "smtp_server"),
        ("emailsender", "smtp_username"),
        ("emailsender", "smtp_password"),
    ],
    "/etc/squirro/filtering.ini": [
        ("filtering", "db"),
    ],
    "/etc/squirro/fingerlogging.info.ini": [("fingerlogging.info", "db")],
    "/etc/squirro/fingerprint.ini": [("fingerprint", "db")],
    "/etc/squirro/frontend.ini": [
        ("flask_app", "secret_key"),
        ("studio", "db"),
    ],
    "/etc/squirro/machinelearning.ini": [
        ("machinelearning", "db"),
    ],
    "/etc/squirro/notes.ini": [("notes", "db")],
    "/etc/squirro/plumber.ini": [("plumber", "db")],
    "/etc/squirro/scheduler.ini": [("scheduler", "db")],
    "/etc/squirro/topic.ini": [("topic", "db")],
    "/etc/squirro/topicproxy.ini": [
        ("topicproxy", "db"),
    ],
    "/etc/squirro/trends.ini": [("trends", "db")],
    "/etc/squirro/user.ini": [("user", "db")],
    "/etc/squirro/userproxy.ini": [("userproxy", "db")],
}


def load_key(key_path: str) -> str:
    """Load key from file if exists, else create a new key.

    Args:
        key_path: Path to key file.

    Returns:
        Key
    """
    key_fp, exist = check_file_exists(key_path)
    if exist:
        key = key_fp.read_text("utf-8")
        logging.info("Using existing key from %s", key_fp)
    else:
        key_fp.parent.mkdir(parents=True, exist_ok=True)
        key = fernet.Fernet.generate_key().decode()
        key_fp.write_text(key, "utf-8")
        logging.info("Writing key to %s", key_fp)
    return key


def main() -> None:
    """Entrypoint."""
    key = load_key("/etc/squirro/.squirro.key")
    vault_path = "/etc/squirro/.vault.json"
    vault_fp, exist = check_file_exists(vault_path)
    vault_fp.parent.mkdir(parents=True, exist_ok=True)
    vault: dict[str, dict[str, str]] = (
        json.loads(vault_fp.read_text("utf-8")) if exist else defaultdict(dict)
    )

    for filepath, items in VALUES_TO_ENCRYPT.items():
        path, exist = check_file_exists(filepath)
        if not exist:
            continue
        logging.info("Reading %s", path)

        cfg = ConfigParser()
        cfg.read(path)

        for section, service in items:
            logging.info("Decrypting %s:%s", section, service)
            if not cfg[section][service].startswith("CK_CMD::"):
                continue
            cfg[section][service] = decrypt(vault[section][service], key)

        logging.info("Writing %s", path)
        with path.open("w", encoding="utf-8") as fp:
            cfg.write(fp)


if __name__ == "__main__":
    main()
