"""Update system configs."""

import json
import logging
from configparser import ConfigParser
from typing import TYPE_CHECKING

from octopus.utils import ensure_file_exists, load_config, set_log_verbosity

if TYPE_CHECKING:
    from pathlib import Path


set_log_verbosity(logging.INFO)


def _update_ini(src: "Path", dst: "Path") -> None:
    """Update destination .ini file with source .ini file."""
    cfg = load_config()
    src_config, dst_config = ConfigParser(), ConfigParser()
    src_config.read(src)
    dst_config.read(dst)

    for section, options in src_config.items():
        if section == "DEFAULT":
            continue
        if not dst_config.has_section(section):
            dst_config.add_section(section)
        for option, value in options.items():
            dst_config.set(section, option, value)

    if "emailsender" in dst_config.sections():
        # Internal server requires smtp_login to be true while OCBC is false
        dst_config.set(
            "emailsender",
            "smtp_login",
            ("true" if (cfg.get("squirro", "environment") == "INTERNAL") else "false"),
        )

    # Update query_templates project_ids with the project_ids from the config
    if "query_templates" in src_config.sections():
        dst_config["query_templates"]["project_ids"] = cfg.get("squirro", "project_id")

    with dst.open("w", encoding="utf-8") as f:
        dst_config.write(f)


def _update_line_based_matching(src: "Path", dst: "Path") -> None:
    """Update destination file with source file line by line."""
    with dst.open(encoding="utf-8") as dst_file:
        dst_dict = {line.split()[0]: line for line in dst_file}

    with src.open(encoding="utf-8") as src_file:
        for src_line in src_file:
            src_line_first_word = src_line.split()[0]
            dst_dict[src_line_first_word] = (
                src_line  # This will replace or add the line
            )

    # Write the updated lines back to the destination file
    with dst.open("w", encoding="utf-8") as dst_file:
        dst_file.writelines(dst_dict.values())


def main() -> None:
    """Entrypoint."""
    mapping_path = ensure_file_exists("mappings.json")
    with mapping_path.open(encoding="utf-8") as fp:
        mapping = json.load(fp)

    logging.info("Updating system configs")
    for src_path, dst_path in mapping.items():
        logging.info("Updating %s with %s", dst_path, src_path)
        src = ensure_file_exists(f"configs/{src_path}")
        dst = ensure_file_exists(dst_path)

        if src.name.endswith(".ini"):
            _update_ini(src, dst)
        else:
            _update_line_based_matching(src, dst)


if __name__ == "__main__":
    main()
