"""Update status tracking config."""

import json
import logging
from pathlib import Path
from typing import TYPE_CHECKING

import requests

from octopus.utils import load_config, set_log_verbosity

if TYPE_CHECKING:
    from configparser import ConfigParser
    from typing import Any


set_log_verbosity(logging.INFO)


def main() -> None:
    """Entrypoint."""
    cfg: ConfigParser = load_config()

    logging.info("Loading status configuration")
    with Path(
        "../../integration/studio/document_status_tracking/status-configuration.json"
    ).open(encoding="utf-8") as f:
        data: dict[str, Any] = json.load(f)

    logging.info("Fetching current version")
    version: int
    try:
        res = requests.get(
            "http://localhost/studio/document_status_tracking/projects/"
            f"{cfg['squirro']['project_id']}/config?token={cfg['squirro']['token']}",
            timeout=60,
        )
        res.raise_for_status()
        version = res.json()["config"]["version"]
    except requests.HTTPError:
        logging.warning(
            "Failed to fetch current version. This is likely the first run."
        )
        version = 0
    logging.info("Current version: %s", version)

    version += 1
    res = requests.post(
        "http://localhost/studio/document_status_tracking/projects/"
        f"{cfg['squirro']['project_id']}/config?token={cfg['squirro']['token']}",
        json={
            "version": version,
            "project_id": cfg["squirro"]["project_id"],
            "status_map": data,
        },
        headers={"Content-Type": "application/json"},
        timeout=60,
    )
    res.raise_for_status()
    logging.info("Updated status configuration to version %s", version)


if __name__ == "__main__":
    main()
