"""Update Squirro configurations."""

import json
import logging
from pathlib import Path
from typing import TYPE_CHECKING

from octopus.clients.squirro_client import init_squirro_client
from octopus.utils import load_config, set_log_verbosity

if TYPE_CHECKING:
    from typing import Any

    from squirro_client import SquirroClient


set_log_verbosity(logging.INFO)

configs = {
    "activity_configurations.json": "project",
    "cluster_configurations.json": "server",
    "octopus_configurations.json": "project",
}

# handle conflict between deployment configs and user-modified configs in the ui by
# only updating the configs in overrides if they already exist
overrides = {
    "octopus_configurations.json": [
        "topic.item.allowed-reader-edit-keywords",
        "topic.typeahead.facetvalue.configuration",
        "topic.typeahead.popular.scope",
        "topic.references",
        "frontend.notes.enabled",
        "frontend.userapp.help-dropdown",
    ],
    "cluster_configurations.json": [
        "frontend.userapp.use-react-widgets",
        "security.expiring-sessions",
    ],
}


def _read_config(path: str) -> "dict[str, Any]":
    """Load config.

    Args:
        path: The path to the config file

    Returns:
        The config
    """
    fp = Path("configs") / path
    with fp.open() as f:
        config: dict[str, Any] = json.load(f)
        return config


def main() -> None:
    """Entrypoint."""
    logging.info("Updating cluster configurations")
    sq_client: SquirroClient
    sq_client, _ = init_squirro_client()

    cfg = load_config()
    project_ids = {
        "activity": cfg["activity"]["project_id"],
        "octopus": cfg["squirro"]["project_id"],
    }

    for path, scope in configs.items():
        logging.info("Updating %s configurations from `%s`", scope, path)

        override_keys = overrides.get(path, [])

        config = _read_config(path)
        if scope == "project":
            project_id = project_ids[path.split("_", 1)[0]]
            config_before_update = sq_client.get_project_configuration(project_id)[
                "config"
            ]
            for k, v in config.items():
                if k in override_keys or k not in config_before_update:
                    sq_client.set_project_configuration(project_id, k, v)
        else:
            config_before_update = sq_client.get_server_configuration()["config"]
            for k, v in config.items():
                if k in override_keys or k not in config_before_update:
                    sq_client.set_server_configuration(k, v)


if __name__ == "__main__":
    main()
