"""Set group permissions in Squirro."""

import logging
from typing import TYPE_CHECKING

from octopus.clients import init_squirro_client
from squirro_client import ConflictError

if TYPE_CHECKING:
    from argparse import Namespace


def main(args: "Namespace") -> None:
    """Entrypoint.

    Args:
        args: The arguments

    Raises:
        ValueError: If the group does not exist
    """
    logging.info("Creating user group")
    sq_client, project_id = init_squirro_client()

    group_id = ""
    for single_group_info in sq_client.get_groups():
        if single_group_info["name"] == args.group:
            group_id = single_group_info["id"]
            break
    else:
        msg = f"Group {args.group} does not exists"
        logging.error(msg)
        raise ValueError(msg)

    logging.warning("Updating project member")
    try:
        sq_client.add_project_member(project_id, args.role, group_id=group_id)
    except ConflictError:
        logging.info("Group already has a role in the project.")

    logging.info("Group permissions updated successfully.")


if __name__ == "__main__":
    import argparse

    parser = argparse.ArgumentParser()
    parser.add_argument(
        "--group", type=str, required=True, help="User group name to be created"
    )
    parser.add_argument(
        "--role",
        type=str,
        choices=["reader", "member", "admin"],
        required=True,
        help="Project role to be set for group.",
    )

    main(parser.parse_args())
