"""Set dashboard permissions in Squirro."""

import logging
from typing import TYPE_CHECKING

from octopus.clients import init_squirro_client
from octopus.utils import set_log_verbosity

if TYPE_CHECKING:
    from argparse import Namespace

set_log_verbosity(logging.INFO)


def main(args: "Namespace") -> None:
    """Entrypoint.

    Args:
        args: The arguments

    Raises:
        ValueError: If the group does not exist
    """
    sq_client, project_id = init_squirro_client()

    group_id = ""
    for single_group_info in sq_client.get_groups():
        if single_group_info["name"] == args.group:
            group_id = single_group_info["id"]
            break
    else:
        msg = f"Group {args.group} does not exists"
        logging.error(msg)
        raise ValueError(msg)

    all_dashboards = sq_client.get_dashboards(project_id)
    for dashboard_name in args.dashboards.split(","):
        name = dashboard_name.strip()
        dashboard_id = ""
        for single_dashboard_info in all_dashboards:
            if single_dashboard_info["title"] == name:
                dashboard_id = single_dashboard_info["id"]
                break
        else:
            msg = f"Dashboard {name} does not exist"
            logging.error(msg)
            raise ValueError(msg)

        logging.info("Adding group %s to dashboard %s", args.group, name)

        permissions_list = sq_client.get_dashboard_permissions(
            project_id, dashboard_id
        )["permissions"]

        is_group_in_dashboard = False
        for permission in permissions_list:
            if permission.get("group_id") == group_id:
                logging.info(
                    "Group %s already exists in dashboard %s permissions list",
                    args.group,
                    name,
                )
                is_group_in_dashboard = True
                break

        if is_group_in_dashboard:
            continue

        permissions_list.append(
            {
                "group_id": group_id,
                "permission": "view",
                "dashboard_id": dashboard_id,
            }
        )
        sq_client.create_dashboard_permissions(
            project_id, dashboard_id, permissions_list
        )

        logging.info("Added group %s to dashboard %s", args.group, name)


if __name__ == "__main__":
    import argparse

    parser = argparse.ArgumentParser()
    parser.add_argument(
        "--dashboards",
        type=str,
        required=True,
        help="Comma-separated string of dashboards for which permissions are updated",
    )
    parser.add_argument(
        "--group",
        type=str,
        required=True,
        help="User group name to add to dashboard permissions",
    )

    main(parser.parse_args())
