# mypy: ignore-errors
import argparse
import json
import logging
import os
import subprocess
import time
from configparser import ConfigParser
from datetime import datetime

import redis
from redis.sentinel import Sentinel

# Script version. It's recommended to increment this with every change, to make
# debugging easier.
VERSION = "1.0.0"

# Define list of squirro services
_SQUIRRO_SERVICES = [
    "sqconfigurationd",
    "sqcontentd",
    "sqdatasourced",
    "sqdigestmailerd",
    "sqemailsenderd",
    "sqfilteringd",
    "sqfingerprintd",
    "sqfrontendd",
    "sqingesterd",
    "sqmachinelearningd",
    "sqnotesd",
    "sqpdfconversiond",
    "sqplumberd",
    "sqproviderd",
    "sqrelatedstoryd",
    "sqschedulerd",
    "sqsearchd",
    "sqthumblerd",
    "sqtopicd",
    "sqtopicproxyd",
    "sqtrendsd",
    "squserd",
    "squserproxyd",
    "sqwebshotd",
]

# Set up logging.
logging.basicConfig(
    level=logging.INFO, format="%(asctime)s [%(levelname)s] %(message)s"
)
log = logging.getLogger(__name__)
log.setLevel(logging.INFO)


def _setup_config(ini_fp: str) -> "ConfigParser":
    """Read config via configparser.

    Args:
    ---
    ini_fp: str
        Path to ini file.

    Returns:
    -------
    config: configparser.ConfigParser
        ConfigParser object.
    """
    try:
        config = ConfigParser()
        config.read(ini_fp)
    except Exception:
        log.exception(f"Unable to read {ini_fp}")
        raise

    return config


def _restart_service(idx: int, service_name: str, checkfile_fp: str):
    """Restart Squirro service via systemctl.

    Args:
    ----
    idx: int
        The index of service in queue
    service_name: str
        Name of service to restart.
    checkfile_fp: str
        Path to checkfile.
    """
    log.warning(f"\t> Restarting {idx}) {service_name}")
    commands = ["sudo", "systemctl", "restart", service_name]
    log.warning(f"Running command {' '.join(commands)}")
    ret_code = subprocess.call(commands)

    if ret_code:
        log.error(f"Unable to restart {service_name} ")
        return

    try:
        with open(checkfile_fp, "w") as f:
            f.write("reset")
    except Exception:
        log.exception(f"Unable to write to {checkfile_fp}")
        raise


def _reorder_list(items: "list[str]", position: int):
    """Reorder list so services not run at same time.

    Args:
    ----
    items: list
        List of items to reorder.
    position: int
        Position to reorder list.

    Returns:
    -------
    list
        Reordered list.
    """
    length = len(_SQUIRRO_SERVICES)
    return items[length - position :] + items[: length - position]


def write_redis_sentinel_state_file(
    redis_sentinel_state_fp: str, config: "dict"
) -> None:
    log.info("Writing the new config to file.")
    try:
        with open(redis_sentinel_state_fp, "w") as f:
            json.dump(config, f, indent=2)
    except Exception:
        log.exception(f"Unable to write to {redis_sentinel_state_fp}")
        raise

    # Update permissions for file, ensuring it can be read by Squirro
    os.chmod(redis_sentinel_state_fp, 0o777)


def manage_master_host_config(
    redis_sentinel_state_fp: str,
    redis_host: str,
    redis_cache_host: str,
    master_changed_redis_server: bool = False,
    master_changed_redis_server_cache: bool = False,
) -> "dict":  # type: ignore[type-arg]
    """Set master hosts for redis server and redis server cache.

    Args:
    ----
    redis_sentinel_state_fp: str
        Path to redis sentinel state file.
    redis_host: str
        Redis host.
    redis_cache_host: str
        Redis cache host.
    master_changed_redis_server: bool
        Whether master redis server has changed.
    master_changed_redis_server_cache: bool
        Whether master redis server cache has changed.

    Returns:
    -------
    dict
        Config dict.
    """
    timestamp = datetime.now().strftime("%Y-%m-%dT%H:%M:%S")  # Declare timestamp

    if os.path.exists(redis_sentinel_state_fp):
        log.info("Redis sentinel state file exists. Loading config...")
        try:
            with open(redis_sentinel_state_fp) as f:
                config = json.load(f)
        except Exception:
            log.exception(f"Unable to read {redis_sentinel_state_fp}")
            raise

        if master_changed_redis_server:
            log.info("master_changed_redis_server has changed. Updating...")
            config["master_redis_server_host"] = redis_host
            config["master_redis_server_ts"] = timestamp

        if master_changed_redis_server_cache:
            log.info("master_changed_redis_server_cache has changed. Updating...")
            config["master_redis_server_cache_host"] = redis_cache_host
            config["master_redis_server_cache_ts"] = timestamp
    else:
        log.info("Redis sentinel state file doesn't exist. Generating config...")
        config = {
            "master_redis_server_host": redis_host,
            "master_redis_server_ts": timestamp,
            "master_redis_server_cache_host": redis_cache_host,
            "master_redis_server_cache_ts": timestamp,
        }
        write_redis_sentinel_state_file(redis_sentinel_state_fp, config)

    if master_changed_redis_server or master_changed_redis_server_cache:
        write_redis_sentinel_state_file(redis_sentinel_state_fp, config)

    return config


def check_config_field_changed(
    config: "dict",  # type: ignore[type-arg]
    key: str,
    curr_value: "str | int",
) -> bool:
    config_value = config[key]
    changed = config_value != curr_value
    if changed:
        log.warning(f"{key} changed from {config_value} to {curr_value}")
    return changed


def main(config):
    """The main method.

    Any exceptions are caught outside of this method and will be
    handled.
    """
    # TODO add validation
    # Sentinel config (applies to both redis server and redis server cache)
    redis_sentinel_state_fp = config.get("sentinel", "redis_sentinel_state_fp")
    hosts = config.get("sentinel", "hosts").split(",")
    cooloff_time = int(config.get("sentinel", "cooloff_time"))
    server_order = int(config.get("sentinel", "order"))
    sentinel_port = int(config.get("sentinel", "port"))

    # Redis server config
    redis_server_master_name = config.get("redis-server", "master_name")
    redis_server_cache_master_name = config.get("redis-server-cache", "master_name")

    # Connect to Redis Sentinel
    log.warning("Connecting to Redis Sentinel")
    try:
        sentinel = Sentinel(
            [(host, sentinel_port) for host in hosts],
            socket_timeout=1,
        )
    except Exception:
        log.exception("Unable to connect to Redis Sentinel")
        raise

    # Reorder list of services to ensure same services not reset at same time
    sq_services_to_reset = _reorder_list(_SQUIRRO_SERVICES, server_order)

    while True:
        try:
            # Get the current master's information
            curr_redis_server_master: tuple[str, int] = sentinel.discover_master(
                redis_server_master_name
            )
            curr_redis_server_master_host: str = curr_redis_server_master[0]

            curr_redis_server_cache_master: tuple[str, int] = (
                sentinel.discover_master(redis_server_cache_master_name)
            )
            curr_redis_server_cache_master_host: str = curr_redis_server_cache_master[0]

            # Get the latest host and timestamp of last known master
            master_state_conf = manage_master_host_config(
                redis_sentinel_state_fp,
                curr_redis_server_master_host,
                curr_redis_server_cache_master_host,
            )

            # Determine appropriate message
            master_changed_redis_server = check_config_field_changed(
                master_state_conf,
                "master_redis_server_host",
                curr_redis_server_master_host,
            )
            master_changed_redis_server_cache = check_config_field_changed(
                master_state_conf,
                "master_redis_server_cache_host",
                curr_redis_server_cache_master_host,
            )

            # Go back to sleep if same host
            if (
                master_changed_redis_server or master_changed_redis_server_cache
            ):  # Check if the master has changed
                # If redis server or redis server cache reset services
                log.warning("**** Master Redis Change ****")
                # Update master host config

                manage_master_host_config(
                    redis_sentinel_state_fp,
                    curr_redis_server_master_host,
                    curr_redis_server_cache_master_host,
                    master_changed_redis_server,
                    master_changed_redis_server_cache,
                )

                # Check file to determine whether services are being reset or not
                check_file = "sq_services_restarted.check"

                # If check file doesn't exist it mean services
                if not os.path.exists(check_file):
                    log.warning("Restarting since check file does not exist.")
                    log.warning(
                        f"There are a total of {len(sq_services_to_reset)}"
                        " to be reset."
                    )
                    for idx, service in enumerate(sq_services_to_reset):
                        # Restart Service
                        _restart_service(idx, service, check_file)
                        # Sleep between restarts
                        time.sleep(1)
                    log.error("System failed, service already being reset")
                    time.sleep(60)
                # After services reset, remove checkfile
                if os.path.exists(check_file):
                    log.info("Services done being reset. Removing check file.")
                    try:
                        os.remove(check_file)
                    except Exception:
                        log.exception("Unable to remove check file")

            else:
                log.warning(
                    "Master Redis server host set to "
                    f"{master_state_conf['master_redis_server_host']} "
                    f"at {master_state_conf['master_redis_server_ts']}"
                )
                log.warning(
                    f"Master Redis cache host set to "
                    f"{master_state_conf['master_redis_server_cache_host']} "
                    f"at {master_state_conf['master_redis_server_cache_ts']}"
                )
                log.warning(f"Nothing to do sleeping for {cooloff_time} seconds")
                time.sleep(cooloff_time)
        except redis.sentinel.MasterNotFoundError:
            log.error("Unable to detect master")
        except redis.exceptions.ConnectionError:
            log.exception("Cannot connect to Redis Sentinel.")

        log.warning(f"Nothing to do sleeping for {cooloff_time} seconds")
        time.sleep(cooloff_time)


def run():
    """Main entry point run by __main__ below.

    No need to change this usually.
    """
    # Setup config to read redis config
    config = _setup_config(args.ini_location)

    log.info("Starting process (version %s).", VERSION)
    log.debug("Arguments: %r", args)
    log.debug("Config: %r", args)

    try:
        main(config)
    except Exception:
        log.exception("Processing error")
        raise


# This is run if this script is executed, rather than imported.
if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--version", action="version", version=VERSION)
    parser.add_argument(
        "--verbose", "-v", action="count", help="Show additional information."
    )
    parser.add_argument(
        "--ini-location",
        type=str,
        help="Redis Sentinel Configuration path",
        default="/etc/redis/redis-sentinel-master.ini",
    )
    args = parser.parse_args()
    run()
