"""Project export."""

import ast
import logging
from pathlib import Path
from typing import TYPE_CHECKING

from octopus.clients import init_squirro_client
from octopus.utils import load_config, set_log_verbosity

if TYPE_CHECKING:
    from typing import Any


set_log_verbosity(logging.INFO)


def main(project: str) -> None:
    """Entrypoint."""
    logging.info("Exporting %s project", project)

    cfg = load_config("../../config/main.ini")
    sq_client, project_id = init_squirro_client(
        cfg, project="squirro" if project == "octopus" else project
    )

    project_export: dict[str, Any] = sq_client.export_project(
        project_id,
        export_ais_published_models=False,
        export_app_nav_bar=False,
        export_communities=False,
        export_community_types=False,
        export_dashboard_loaders=True,
        export_dashboard_widgets=False,
        export_dashboards=True,
        export_default_sort=False,
        export_email_templates=False,
        export_enrichment_pipelets=False,
        export_enrichments=False,
        export_facets=False,
        export_guide_file=False,
        export_items=False,
        export_items_config=None,
        export_machinelearning_jobs=False,
        export_machinelearning_models=False,
        export_machinelearning_workflows=False,
        export_pipeline_workflows=False,
        export_project_configuration=False,
        export_project_picture=True,
        export_project_translations=False,
        export_smartfilters=False,
        export_sources=False,
        export_sources_dataloader_plugins=False,
        export_synonyms=True,
        export_trends=False,
    )
    export_dict: dict[str, Any] = ast.literal_eval(
        project_export["data"].decode("utf-8")
    )

    while True:
        res = sq_client.get_project_export(project_id, export_dict["export_id"])
        if res.get("data"):
            break

    with Path(f"../../assets/project_exports/{project.lower()}.tar.gz").open("wb") as f:
        f.write(res["data"])

    logging.info("Project export done")


if __name__ == "__main__":
    import argparse

    parser = argparse.ArgumentParser()
    parser.add_argument(
        "--project",
        default="octopus",
        type=str,
        choices=["octopus", "activity"],
        help="Project to export",
    )
    args = parser.parse_args()
    main(args.project)
