"""Create labels."""

import logging
from typing import TYPE_CHECKING

from labels import monitoring_labels, octopus_labels

from octopus.clients import init_squirro_client
from octopus.utils import load_config, set_log_verbosity
from squirro_client import ConflictError

if TYPE_CHECKING:
    from squirro_client import SquirroClient

set_log_verbosity(logging.INFO)


def main() -> None:
    """Entrypoint.

    Raises:
        ValueError: If the label data type and display name are not provided
    """
    sq_client: SquirroClient
    project_id: str
    sq_client, project_id = init_squirro_client()
    monitoring_project_id: str = load_config()["activity"]["project_id"]

    for pid, labels in zip(
        (project_id, monitoring_project_id),
        (octopus_labels, monitoring_labels),
        strict=False,
    ):
        for label in labels:
            label_analysed = label.get("analysed", False)
            label_data_type = label.get("data_type", "string")
            label_display_name = label.get("display_name", None)
            label_format_instr = None
            label_groupname = None
            label_name = label.get("name", None)
            label_searchable = label.get("searchable", False)
            label_synonyms_id = None
            label_typeahead = label.get("typeahead", False)
            label_visible = label.get("visible", False)

            if not label_data_type or not label_display_name:
                msg = "Label data type and display name are required."
                logging.error(msg)
                raise ValueError(msg)

            try:
                logging.info("Creating label: %s", label_name)
                sq_client.new_facet(
                    pid,
                    label_name,
                    data_type=label_data_type,
                    display_name=label_display_name,
                    group_name=label_groupname,
                    visible=label_visible,
                    format_instr=label_format_instr,
                    searchable=label_searchable,
                    typeahead=label_typeahead,
                    analyzed=label_analysed,
                    synonyms_id=label_synonyms_id,
                )
            except ConflictError:
                logging.info("Label already exists: %s", label_name)
            except Exception:  # pylint: disable=broad-except
                logging.exception("Error creating label: %s", label_name)

    logging.info("Labels created successfully.")


if __name__ == "__main__":
    main()
