"""Storage plugin for WFI."""

import logging
from io import BytesIO
from typing import TYPE_CHECKING

from octopus.clients import init_wfi_client
from octopus.utils import load_config, set_log_verbosity
from squirro.lib.storage.container.base import StorageContainer

if TYPE_CHECKING:
    from typing import Any

set_log_verbosity(logging.INFO)


class WFIStorageContainer(StorageContainer):  # type: ignore[misc]
    """Storage container for WFI."""

    def __init__(self, **_: "dict[str, Any]") -> None:
        """Initialize the storage container."""
        super().__init__()
        self.project_id: str = load_config()["squirro"]["project_id"]
        self.wfi_client = init_wfi_client()

    def download(self, doc_id: str) -> bytes:
        """Download the content of the document.

        Args:
            doc_id: Document ID.

        Returns:
            Content of the document.
        """
        content: bytes = self.wfi_client.fetch_content(doc_id.split("/")[1])["content"]
        return content

    def get_external_url(self, doc_id: str) -> str:
        """Get the storage URL.

        Args:
            doc_id: Document ID.

        Returns:
            URL of the storage.
        """
        return f"/studio/storage/wfi/{doc_id}"

    def open(self, doc_id: str) -> BytesIO:
        """Open the storage URL.

        Args:
            doc_id: Document ID.

        Returns:
            BytesIO object.
        """
        content: bytes = self.download(doc_id)
        bytes_io = BytesIO(content)
        bytes_io.seek(0)
        return bytes_io
