# mypy: ignore-errors
import html
import json
import logging
import os
import urllib.parse
from typing import Dict, Generator, List

from flask import jsonify, redirect, request, session, url_for
from flask_dance.contrib.azure import azure, make_azure_blueprint
from oauthlib.oauth2 import TokenExpiredError

from squirro.common.dependency import get_injected
from squirro.dataloader.auth import auth_tools
from squirro.sdk.studio import DataloaderFrontendPlugin

config = get_injected("config")
auth_tools.configure_oauth2_lib(config)

TARGET_NAME = "Microsoft Exchange"
log = logging.getLogger(__name__)
plugin = DataloaderFrontendPlugin(__name__)


def _graph_scopes(scopes: List[str]) -> Generator[str, None, None]:
    return (f"https://graph.microsoft.com/{s}" for s in scopes)


# Azure will drop "offline" and unavailable work-account scopes
os.environ["OAUTHLIB_RELAX_TOKEN_SCOPE"] = "1"  # nosec: B105

token_expected_scopes = set(_graph_scopes(["User.Read", "Mail.Read"]))

client_id = config.get("dataloader", "exchange_client_id", fallback=None)
client_secret = config.get("dataloader", "exchange_client_secret", fallback=None)
tenant_id = config.get("dataloader", "exchange_tenant_id", fallback=None)

if not client_id or not client_secret or not tenant_id:
    log.warning("Client keys are missing in %s plugin", TARGET_NAME)

auth_blueprint = make_azure_blueprint(
    client_id=client_id,
    client_secret=client_secret,
    scope=["offline_access"] + sorted(token_expected_scopes),
    tenant=tenant_id,
    redirect_to="done",
    login_url="/login",
    session_class=auth_tools.flask_dance_session_factory(config),
)
plugin.register_blueprint(auth_blueprint, url_prefix="/pl")
oauth = azure


def get_account_id(token: Dict[str, str]) -> str:
    oauth.token = token
    resp = oauth.get("/v1.0/me")
    resp.raise_for_status()
    return resp.json()["userPrincipalName"]


# Common parts:
def token_delete() -> None:
    """Clear flask dance token storage."""
    try:
        del auth_blueprint.token  # Delete OAuth token from storage
    except KeyError:  # token wasn't set
        pass


@plugin.route("/")
def start():
    session["next_url"] = auth_tools.assert_valid_next_url(request.args["back_to"])
    login_url = url_for(f"{auth_blueprint.name}.login")
    log.info("Redirecting to %r with next URL %r", login_url, session["next_url"])
    return redirect(login_url)


@plugin.route("/done")
def done():
    next_url = session["next_url"]
    start_url = f"{url_for('start')}?{urllib.parse.urlencode({'back_to': next_url})}"
    token = oauth.token  # populated by flask-dance
    if not token:
        log.warning("OAuth token not acquired, redirecting to OAuth `start`")
        return redirect(start_url)

    log.info("Token scope: %s", sorted(token["scope"]))
    token_as_str = _pack_token(token)
    log.info("Redirecting to %r with token of len %d", next_url, len(token_as_str))
    next_url = f"{next_url}?{urllib.parse.urlencode({'token': token_as_str})}"
    return redirect(next_url)


def _pack_token(token: dict) -> str:
    return json.dumps(token, separators=(",", ":"), sort_keys=True)


@plugin.route("/verify", methods=["POST"])
def verify_token():
    token: "Dict[str, str]" = request.json["token"]
    try:
        login = get_account_id(token)
    except TokenExpiredError:
        log.warning("Token expired")
        token_delete()
        raise
    log.info("Successful login with %s", TARGET_NAME)
    return jsonify(
        {"message": f"You are logged into {TARGET_NAME} as {html.escape(login)}."}
    )
