const overrides = {
  ...Overrides.PredQuery,
  VerticalTab: Resources.resolved["VerticalTab.js"],
};

export default (props) => {
  const [activeFiltersTab, setActiveFiltersTab] = React.useState(null);
  const [filtersAnchorEl, setFiltersAnchorEl] = React.useState(null);
  const isFiltersOpen = Boolean(filtersAnchorEl);

  const [documentDateQueries, setDocumentDateQueries] = React.useState([]);
  const [dateMode, setDateMode] = React.useState("at");
  const [dateView, setDateView] = React.useState("day");

  const [prevFilter, setPrevFilter] = React.useState({});
  const [activeQueries, setActiveQueries] = React.useState({
    company_name: null,
    document_type: null,
    document_date: null,
    cso_name: null,
    rm_name: null,
  });

  const prevQueries = Hooks.usePrevious(activeQueries);

  const { addFacetValue, removeFacetValue } = Hooks.useDrilldown(
    Globals.dashboardState,
    props.widgetModel,
  );

  React.useEffect(() => {
    const labelChanged = Object.entries(activeQueries).find(
      ([label, value]) => value && value !== prevQueries[label],
    );

    if (labelChanged) {
      const [labelName, labelValue] = labelChanged;

      labelName in prevQueries &&
        removeFacetValue(labelName, prevQueries.labelName);
      addFacetValue(labelName, labelValue);
    } else {
      if (!_.isEmpty(prevFilter)) {
        removeFacetValue(...prevFilter);
        addFacetValue(...prevFilter);
      }
    }
  }, [activeQueries]);

  const DocumentDateTab = ({
    dateMode: mode,
    dateView: view,
    setDateMode,
    setDateView,
    handleClose,
    documentDateQueries,
    setDocumentDateQueries,
  }) => {
    const handleAccept = (date, viewArg) => {
      const newDate = date.format(viewArg === "day" ? "YYYY-MM-DD" : "YYYY-MM");

      switch (mode) {
        case "from":
        case "to":
          const incomingQuery = {
            type: "query",
            mode: mode,
            value:
              mode === "to"
                ? `document_date<${newDate}`
                : `document_date>${newDate}`,
          };

          addFacetValue(null, incomingQuery.value);

          if (!documentDateQueries.length) {
            setDocumentDateQueries([incomingQuery]);
            return;
          }

          if (documentDateQueries[0].mode === "at") {
            documentDateQueries.forEach((query) => {
              removeFacetValue(
                query.type === "facet" ? "document_date" : null,
                query.value,
              );
            });
            setDocumentDateQueries([incomingQuery]);
            return;
          }

          const newDocumentDateQueries = documentDateQueries.filter((item) => {
            item.mode === mode && removeFacetValue(null, item.value);
            return !(item.mode === mode);
          });
          newDocumentDateQueries.push(incomingQuery);
          setDocumentDateQueries(newDocumentDateQueries);

          break;

        case "at":
          documentDateQueries.forEach((query) =>
            removeFacetValue(
              query.type === "facet" ? "document_date" : null,
              query.value,
            ),
          );

          if (viewArg === "day") {
            setDocumentDateQueries([
              {
                type: "facet",
                mode: "at",
                value: newDate,
              },
            ]);
            addFacetValue("document_date", newDate);
            return;
          }

          const startOfMonth = date.set("date", 1);
          const startOfNextMonth = moment(startOfMonth).add(1, "M");

          const valueStart = `document_date>${startOfMonth.format(
            "YYYY-MM-DD",
          )}`;
          const valueEnd = `document_date<${startOfNextMonth.format(
            "YYYY-MM-DD",
          )}`;

          setDocumentDateQueries([
            {
              type: "query",
              mode: "at",
              value: `${valueStart} ${valueEnd}`,
            },
          ]);
          addFacetValue(null, `${valueStart} ${valueEnd}`);
      }
      handleClose();
    };

    const renderOptions = () => {
      const toggleButtonStyles = (theme) => ({
        "& .Mui-selected": {
          backgroundColor: `${theme.palette.primary.main} !important`,
          color: `${theme.palette.primary.contrastText} !important`,
        },
        "& .Mui-selected:hover": {
          backgroundColor: "red",
        },
      });

      return (
        <MUI.Stack
          direction="row"
          justifyContent="space-between"
          sx={{ px: 2, pt: 2 }}
        >
          <MUI.ToggleButtonGroup
            sx={toggleButtonStyles}
            size="small"
            value={mode}
            exclusive
            onChange={(e, newValue) => {
              newValue && setDateMode(newValue);
            }}
          >
            <MUI.ToggleButton value="at">At</MUI.ToggleButton>
            <MUI.ToggleButton value="from">From</MUI.ToggleButton>
            <MUI.ToggleButton value="to">To</MUI.ToggleButton>
          </MUI.ToggleButtonGroup>

          <MUI.ToggleButtonGroup
            sx={toggleButtonStyles}
            size="small"
            value={view}
            exclusive
            onChange={(e, newValue) => {
              newValue && setDateView(newValue);
            }}
          >
            <MUI.ToggleButton value="day">Day</MUI.ToggleButton>
            <MUI.ToggleButton value="month">Month</MUI.ToggleButton>
          </MUI.ToggleButtonGroup>
        </MUI.Stack>
      );
    };

    const DatePicker = ({ view }) => {
      return (
        <MUIXDatePicker.LocalizationProvider dateAdapter={MUIXAdapterMoment}>
          <MUIXDatePicker.StaticDatePicker
            closeOnSelect
            label="Document Date"
            views={["year", view]}
            openTo={view}
            value={null}
            onAccept={(newDate) => handleAccept(newDate, view)}
            onClose={handleClose}
            onChange={() => {}}
            disableFuture={true}
            slots={{ actionBar: () => null, toolbar: () => null }}
          />
        </MUIXDatePicker.LocalizationProvider>
      );
    };

    return (
      <div>
        {renderOptions()}
        <DatePicker view={view} />
      </div>
    );
  };

  const tabMapping = {
    "Company Name": {
      labelName: "company_name",
      autoCompleteUrl: `/studio/autocomplete_api/company_names/${props.project.id}?q=`,
    },
    "Document Type": {
      labelName: "document_type",
      autoCompleteUrl: `/studio/autocomplete_api/document_types/${props.project.id}`,
    },
    "CSO Name": {
      labelName: "cso_name",
      autoCompleteUrl: `/studio/autocomplete_api/cso_names/${props.project.id}?q=`,
    },
    "RM Name": {
      labelName: "rm_name",
      autoCompleteUrl: `/studio/autocomplete_api/rm_names/${props.project.id}?q=`,
    },
    "Current Status": {
      labelName: "current_doc_status_header",
      autoCompleteUrl: `/studio/document_status_tracking/projects/${props.project.id}/config`,
    },
  };

  const handleClose = () => {
    setFiltersAnchorEl(null);
  };

  const openFilters = (event, filtersTab) => {
    setFiltersAnchorEl(event.currentTarget);
    setActiveFiltersTab(filtersTab);
  };

  return (
    <WidgetContexts.PredQuery.Provider
      value={{
        project_id: props.project.id,
        openFilters,
        addFacetValue,
        removeFacetValue,
      }}
    >
      <MUI.Popover
        open={isFiltersOpen}
        anchorEl={filtersAnchorEl}
        onClose={handleClose}
        anchorOrigin={{
          vertical: "bottom",
          horizontal: "centre",
        }}
        PaperProps={{ sx: paperStyles }}
      >
        {activeFiltersTab === "Document Date" ? (
          <DocumentDateTab
            handleClose={handleClose}
            dateMode={dateMode}
            dateView={dateView}
            setDateMode={setDateMode}
            setDateView={setDateView}
            documentDateQueries={documentDateQueries}
            setDocumentDateQueries={setDocumentDateQueries}
          />
        ) : (
          <FilterOptions
            label={activeFiltersTab}
            onChange={(e, val) => {
              const { labelName } = tabMapping[activeFiltersTab];
              setActiveQueries({
                ...activeQueries,
                [labelName]: val.name ?? val,
              });
              setPrevFilter([labelName, val.name ?? val]);
              handleClose();
            }}
            autoCompleteAPIUrl={tabMapping[activeFiltersTab]?.autoCompleteUrl}
            fetchOptionsOnInputChange={[
              "Company Name",
              "CSO Name",
              "RM Name",
            ].includes(activeFiltersTab)}
            getOptionLabel={(option) => option.name ?? option}
          />
        )}
      </MUI.Popover>
      <Widgets.PredQuery {...props} overrides={overrides} />
    </WidgetContexts.PredQuery.Provider>
  );
};

const FilterOptions = ({
  label,
  onChange,
  autoCompleteAPIUrl,
  fetchOptionsOnInputChange,
  ...props
}) => {
  const [loading, setLoading] = React.useState(false);
  const [options, setOptions] = React.useState([]);

  React.useEffect(() => {
    if (!open && fetchOptionsOnInputChange) {
      setOptions([]);
    } else if (open && !options.length) {
      setLoading(true);
    }
  }, [open]);

  React.useEffect(() => {
    if (!loading) return undefined;

    fetch(autoCompleteAPIUrl)
      .then((res) => res.json())
      .then((data) => {
        label === "Current Status"
          ? setOptions(
              data.config.status_map.statuses
                .map(({ header }) => header)
                .sort(),
            )
          : setOptions(data);
      })
      .catch(() => setOptions([]))
      .finally(() => setLoading(false));
  }, [loading]);

  let timer;
  const handleInputChange = (event, value) => {
    if (timer) clearTimeout(timer);
    timer = setTimeout(() => {
      fetch(autoCompleteAPIUrl + value)
        .then((res) => res.json())
        .then((data) => setOptions(data))
        .catch(() => setOptions([]));
    }, 300);
  };

  return (
    <MUI.Autocomplete
      open
      loading={loading}
      autoHighlight
      disablePortal
      clearOnBlur={false}
      clearOnEscape={true}
      options={options}
      forcePopupIcon={false}
      PopperComponent="div"
      PaperComponent="div"
      onChange={onChange}
      renderInput={(params) => (
        <div ref={params.InputProps.ref}>
          <InputField
            autoFocus
            {...params.inputProps}
            type="text"
            placeholder={label}
          />
        </div>
      )}
      {...(fetchOptionsOnInputChange && {
        filterOptions: (x) => x,
        onInputChange: handleInputChange,
      })}
      {...props}
    />
  );
};

const paperStyles = (theme) => ({
  display: "flex",
  flexDirection: "column",
  minWidth: "300px",
  //   ...(tabName !== "Document Date" && { height: "50vh" }),
  height: "50vh",
  "& .MuiAutocomplete-popper": {
    position: "static",
    flex: 1,
    overflowY: "scroll",
    "& .MuiAutocomplete-listbox": {
      height: "100%",
      maxHeight: "100%",
    },
  },
  "& .MuiCalendarPicker-root": {
    "& button": {
      color: "unset",
      "&.Mui-selected": {
        color: "#fff",
        "&:hover": {
          backgroundColor: theme.palette.primary.main,
        },
      },
    },
    "& .MuiMonthPicker-root button": {
      my: 1,
      "&:not(.Mui-selected):hover": {
        backgroundColor: theme.palette.hover,
      },
    },
    "& .MuiYearPicker-root button": {
      my: 1,
      "&:not(.Mui-selected):hover": {
        backgroundColor: theme.palette.hover,
      },
    },
  },
});

const InputField = styled("input")(({ theme }) => ({
  padding: `${theme.spacing(1)} ${theme.spacing(2)} !important`,
}));
