export default (props) => {
  const [files, setFiles] = React.useState([]);
  const fileInputRef = React.useRef(null);

  const [companyNames, setCompanyNames] = React.useState([]);
  const [documentType, setDocumentType] = React.useState(null);
  const [documentFormat, setDocumentFormat] = React.useState(null);
  const [documentDate, setDocumentDate] = React.useState(null);
  const [references, setReferences] = React.useState([]);

  const [buttonDisabled, setButtonDisabled] = React.useState(false);
  const [loading, setLoading] = React.useState(false);
  const [loadingProgress, setLoadingProgress] = React.useState(0);
  const [formDisabled, setFormDisabled] = React.useState(false);
  const [snackbarConfig, setSnackbarConfig] = React.useState({
    open: false,
    message: "",
    severity: "",
  });

  const userEmail = props.user.get("email");
  const userName = props.user.get("fullName");
  const userRoles = props.user.get("user_information")?.role_ocbc;

  const showSnackbar = (message, severity) => {
    setSnackbarConfig({
      message,
      open: true,
      severity,
    });
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    setButtonDisabled(true);
    setFormDisabled(true);
    setLoading(true);

    const labels = {};

    const formData = new FormData();
    for (const file of files) {
      formData.append("documents", file);
    }
    if (documentType) labels.document_type_true = [documentType];
    if (documentFormat) labels.document_format = [documentFormat];
    if (companyNames.length > 0) labels.company_name_true = [...companyNames];
    if (documentDate)
      labels.document_date_true = [documentDate.format("YYYY-MM-DD")];
    if (references.length) labels.references = references;
    if (userEmail) labels.user_email = [userEmail];
    if (userName) labels.user_name = [userName];

    formData.append("labels", JSON.stringify(labels));
    formData.append("project_id", props.project.id);

    $.ajax({
      url: `/studio/documents/`,
      type: "POST",
      xhr: function () {
        const myXhr = $.ajaxSettings.xhr();
        if (myXhr.upload) {
          myXhr.upload.addEventListener("progress", updateProgress, false);
        }
        return myXhr;
      },
      success: () => {
        setSnackbarConfig({
          open: true,
          message: `${files.length} ${
            files.length > 1 ? "files have" : "file has"
          } been queued for processing`,
          severity: "success",
        });
      },
      error: () => {
        setSnackbarConfig({
          open: true,
          message:
            "There was a problem while trying to upload the file. Please refresh the page and try again",
          severity: "error",
        });
      },
      complete: () => {
        setFiles([]);
        setDocumentType(null);
        setDocumentFormat(null);
        setCompanyNames([]);
        setDocumentDate(null);
        setReferences([]);
        setButtonDisabled(false);
        setFormDisabled(false);
        setLoadingProgress(0);
        setLoading(false);
        fileInputRef.current.value = null;
      },
      data: formData,
      cache: false,
      contentType: false,
      processData: false,
    });
  };

  const updateProgress = (e) => {
    setLoadingProgress(Math.floor((e.loaded / e.total) * 100));
  };

  return (
    <div>
      <MUI.Snackbar
        open={snackbarConfig.open}
        TransitionComponent={SlideTransition}
        anchorOrigin={{ vertical: "top", horizontal: "center" }}
        autoHideDuration={3000}
        onClose={() =>
          setSnackbarConfig({
            ...snackbarConfig,
            open: false,
          })
        }
      >
        <MUI.Alert
          onClose={() =>
            setSnackbarConfig({
              ...snackbarConfig,
              open: false,
            })
          }
          severity={snackbarConfig.severity}
          variant="filled"
          sx={{ width: "100%" }}
        >
          {snackbarConfig.message}
        </MUI.Alert>
      </MUI.Snackbar>
      <Form onSubmit={handleSubmit}>
        {/* Drag Drop File Upload */}
        <DragDropZone
          userRoles={userRoles}
          files={files}
          setFiles={setFiles}
          fileInputRef={fileInputRef}
          disabled={formDisabled}
          showSnackbar={showSnackbar}
        />
        {files.length > 0 && (
          <MUI.Box
            sx={{
              flex: 1,
              display: "flex",
              flexDirection: "column",
              width: "100%",
              rowGap: "1rem",
              overflowY: "auto",
            }}
          >
            <SelectedFiles files={files} setFiles={setFiles} />
            <LabelsInputFields
              files={files}
              companyNames={companyNames}
              setCompanyNames={setCompanyNames}
              documentType={documentType}
              setDocumentType={setDocumentType}
              documentFormat={documentFormat}
              setDocumentFormat={setDocumentFormat}
              documentDate={documentDate}
              setDocumentDate={setDocumentDate}
              references={references}
              setReferences={setReferences}
              formDisabled={formDisabled}
              setFormDisabled={setFormDisabled}
              buttonDisabled={buttonDisabled}
              loading={loading}
              loadingProgress={loadingProgress}
              projectId={props.project.id}
            />
          </MUI.Box>
        )}
      </Form>
    </div>
  );
};

const DragDropZone = ({
  userRoles,
  files,
  setFiles,
  fileInputRef,
  disabled,
  showSnackbar,
}) => {
  const maxNumFiles = 10;
  const [dragActive, setDragActive] = React.useState(false);

  const handleDrag = (e) => {
    e.preventDefault();
    e.stopPropagation();
    if (e.type === "dragenter" || e.type === "dragover") {
      setDragActive(true);
    } else if (e.type === "dragleave") {
      setDragActive(false);
    }
  };

  const handleDrop = (e) => {
    e.preventDefault();
    e.stopPropagation();
    setDragActive(false);

    if (e.dataTransfer.files && e.dataTransfer.files.length) {
      const fileList = e.dataTransfer.files;
      if (isUploadAllowed(userRoles) && checkFiles(fileList)) {
        setFiles(Array.from(fileList));
      }
    }
  };

  const handleFileChange = (e) => {
    e.preventDefault();
    if (e.target.files && e.target.files.length) {
      const fileList = e.target.files;
      if (isUploadAllowed(userRoles) && checkFiles(fileList)) {
        setFiles(Array.from(fileList));
        fileInputRef.current.value = null;
      }
    }
  };

  const checkFiles = (fileList) => {
    if (fileList.length > maxNumFiles) {
      showSnackbar(`Please select no more than ${maxNumFiles} files`, "error");
      return false;
    }

    let totalFilesSize = 0;
    for (let file of fileList) {
      const fileName = file.name;
      const fileExt = `.${fileName.split(".").slice(-1)}`;

      if (!ACCEPTED_FILE_TYPES.includes(fileExt.toLowerCase())) {
        showSnackbar(`File type ${fileExt} is not supported`, "error");
        return false;
      }

      totalFilesSize += file.size;
    }
    if (totalFilesSize > 50e6) {
      showSnackbar(
        "Exceeded maximum file size. Please try again with files smaller than 50MB",
        "error",
      );
      return false;
    }

    return true;
  };

  const isUploadAllowed = (roles) => {
    if (roles.length == 1 && UPLOAD_NOT_ALLOWED_ROLES.includes(roles[0])) {
      // Not a dual role holder
      showSnackbar("You are not allowed to perform this action.", "error");
      return false;
    }
    return true;
  };

  return (
    <MUI.Box
      onDragEnter={handleDrag}
      data-testid="draggable"
      sx={{
        width: "100%",
        height: files.length ? "15%" : "60%",
        minHeight: "15%",
      }}
    >
      <input
        hidden
        type="file"
        name="file"
        multiple
        disabled={disabled}
        id="input-file-upload"
        accept={ACCEPTED_FILE_TYPES.join(",")}
        ref={fileInputRef}
        onChange={handleFileChange}
      />
      <MUI.Box
        sx={{
          position: "relative",
          display: "flex",
          justifyContent: "center",
          alignItems: "center",
          width: "100%",
          height: "100%",
          border: "2px dashed grey",
          backgroundColor: dragActive ? "lightgrey" : "#f8fafc",
          borderRadius: "8px",
        }}
      >
        <MUI.Stack
          direction={files.length ? "row" : "column"}
          spacing={1}
          sx={{
            display: "flex",
            alignItems: "center",
            textAlign: "center",
          }}
        >
          <MUI.Typography variant="h5">Drop your files here or</MUI.Typography>
          <MUI.Button
            sx={{
              m: 1.5,
              mt: 0,
              "&:disabled": {
                backgroundColor: (theme) =>
                  `${theme.palette.primary.main} !important`,
                color: "white !important",
              },
            }}
            disabled={disabled}
            variant="contained"
            onClick={() => fileInputRef.current.click()}
          >
            Browse files
          </MUI.Button>
          <MUI.Typography
            variant="body2"
            sx={{
              position: "absolute",
              right: 0,
              bottom: -25,
            }}
          >
            Maximum {maxNumFiles} files (50MB in total)
          </MUI.Typography>
        </MUI.Stack>
      </MUI.Box>
      {dragActive && (
        <MUI.Box
          onDragEnter={handleDrag}
          onDragLeave={handleDrag}
          onDragOver={handleDrag}
          onDrop={handleDrop}
          sx={{
            position: "absolute",
            width: "100%",
            height: "100%",
            borderRadius: "1rem",
            top: 0,
            bottom: 0,
            left: 0,
            right: 0,
          }}
        ></MUI.Box>
      )}
    </MUI.Box>
  );
};

const SelectedFiles = ({ files, setFiles }) => {
  const removeFile = (fileName) => {
    setFiles(files.filter((file) => file.name !== fileName));
  };

  return (
    <MUI.Box
      sx={{
        width: "100%",
        display: "flex",
        flexDirection: "column",
        rowGap: "0.5rem",
        overflowY: "auto",
      }}
    >
      <MUI.Typography
        sx={{ textAlign: "left", fontWeight: 700 }}
        variant="body1"
        component="h6"
      >
        {files.length} {files.length === 1 ? "File" : "Files"} Selected
      </MUI.Typography>
      <MUI.Box
        sx={{
          flex: 1,
          overflowY: "auto",
          width: "100%",
          maxHeight: "100%",
          borderRadius: "8px",
          border: "1px solid lightgrey",
          textAlign: "left",
        }}
      >
        {Array.from(files).map((file, index) => (
          <MUI.Box
            key={index}
            sx={{
              px: 2,
              py: 0.5,
              display: "flex",
              alignItems: "center",
              justifyContent: "space-between",
              borderBottom:
                index === files.length - 1 ? "none" : "1px solid lightgrey",
            }}
          >
            <MUI.Typography
              sx={{
                overflow: "hidden",
                whiteSpace: "nowrap",
                textOverflow: "ellipsis",
              }}
              variant="body1"
            >
              {file.name}
            </MUI.Typography>
            <MUI.Tooltip title="Remove file">
              <MUI.IconButton
                size="small"
                onClick={() => removeFile(file.name)}
              >
                <MUI.Icon fontSize="small">Clear</MUI.Icon>
              </MUI.IconButton>
            </MUI.Tooltip>
          </MUI.Box>
        ))}
      </MUI.Box>
    </MUI.Box>
  );
};

const LabelsInputFields = ({
  files,
  companyNames,
  setCompanyNames,
  documentType,
  setDocumentType,
  documentFormat,
  setDocumentFormat,
  documentDate,
  setDocumentDate,
  references,
  setReferences,
  formDisabled,
  setFormDisabled,
  buttonDisabled,
  loading,
  loadingProgress,
  projectId,
}) => {
  const { AutocompleteFromAPI, FreeTextMultiple } =
    Resources.resolved["Inputs.js"];

  return (
    <MUI.Box
      sx={{
        display: "flex",
        flexDirection: "column",
        width: "100%",
        rowGap: "0.5rem",
      }}
    >
      <MUI.Typography
        sx={{ textAlign: "left", fontWeight: 700 }}
        variant="body1"
        component="h6"
      >
        Assign Labels
      </MUI.Typography>

      <AutocompleteFromAPI
        label="Company Name(s)"
        multiple
        value={companyNames}
        onChange={(e, newCompanyNames) => {
          setCompanyNames(
            newCompanyNames.map(
              (companyName) => companyName.name ?? companyName,
            ),
          );
        }}
        autoCompleteAPIUrl={`/studio/autocomplete_api/company_names/${projectId}?q=`}
        fetchOptionsOnInputChange
        isOptionEqualToValue={(option, value) => option.name === value}
        renderOption={(props, option) => (
          <li {...props}>
            <MUI.Box
              sx={{
                width: "100%",
                display: "flex",
                justifyContent: "space-between",
                alignItems: "center",
              }}
            >
              <span>{option.name}</span>
              <span style={{ color: "gray" }}>{option.cif}</span>
            </MUI.Box>
          </li>
        )}
      />

      <MUI.FormControl>
        <MUI.InputLabel>Document Format</MUI.InputLabel>
        <MUI.Select
          label="Document Format"
          value={documentFormat}
          onChange={(e) => setDocumentFormat(e.target.value)}
          sx={{ textAlign: "left" }}
        >
          <MUI.MenuItem value="Hardcopy - Original">
            Hardcopy - Original
          </MUI.MenuItem>
          <MUI.MenuItem value="Hardcopy - Photocopy">
            Hardcopy - Photocopy
          </MUI.MenuItem>
          <MUI.MenuItem value="Softcopy">Softcopy</MUI.MenuItem>
        </MUI.Select>
      </MUI.FormControl>

      <AutocompleteFromAPI
        label="Document Type"
        value={documentType}
        onChange={(e, docs) => setDocumentType(docs)}
        autoCompleteAPIUrl={`/studio/autocomplete_api/document_types/${projectId}`}
      />

      <DocumentDatePicker
        documentDate={documentDate}
        setDocumentDate={setDocumentDate}
        disabled={formDisabled}
        projectId={projectId}
      />

      <FreeTextMultiple
        label="References"
        value={references}
        setValue={setReferences}
        setIsFormDisabled={setFormDisabled}
        valueDelimiter=";"
      />

      <MUI.Button
        variant="contained"
        sx={{
          alignSelf: "center",
          position: "relative",
          "&:disabled": {
            backgroundColor: (theme) =>
              buttonDisabled && `${theme.palette.primary.main} !important`,
            color: "white !important",
          },
        }}
        disabled={!files || buttonDisabled || formDisabled}
        type="submit"
      >
        {loading && (
          <MUI.LinearProgress
            sx={{
              position: "absolute",
              top: "0",
              width: "100%",
              height: "100%",
              "&.MuiLinearProgress-colorPrimary": {
                backgroundColor: "rgba(0, 0, 0, 0)",
              },
              "& .MuiLinearProgress-barColorPrimary": {
                backgroundColor: "rgba(0, 0, 0, 0.2)",
              },
              borderRadius: "inherit",
              zIndex: 1,
            }}
            value={loadingProgress}
            variant="determinate"
          />
        )}
        <MUI.Box
          sx={{
            zIndex: 2,
          }}
        >
          {loading ? "SUBMITTING" : "SUBMIT"}
        </MUI.Box>
      </MUI.Button>
    </MUI.Box>
  );
};

const DocumentDatePicker = ({ documentDate, setDocumentDate, disabled }) => {
  return (
    <MUIXDatePicker.LocalizationProvider dateAdapter={MUIXAdapterMoment}>
      <MUIXDatePicker.DatePicker
        label="Document Date"
        disabled={disabled}
        value={documentDate}
        onChange={(date) => {
          setDocumentDate(date);
        }}
        disableFuture={true}
        format="DD/MM/YYYY"
      />
    </MUIXDatePicker.LocalizationProvider>
  );
};

const SlideTransition = (props) => <MUI.Slide {...props} direction="down" />;

const Form = styled("form")({
  display: "flex",
  justifyContent: "flex-start",
  alignItems: "center",
  flexDirection: "column",
  textAlign: "center",
  gap: "1rem",
  height: "100%",
});

const ACCEPTED_FILE_TYPES = [
  ".pdf",
  ".zip",
  ".doc",
  ".docx",
  ".ppt",
  ".pptx",
  ".xls",
  ".xlsx",
  ".xlsm",
  ".txt",
  ".tif",
  ".tiff",
  ".jpeg",
  ".jpg",
  ".png",
  ".msg",
];

const UPLOAD_NOT_ALLOWED_ROLES = ["Workbench:CSSupportChecker"];
