const overrides = {
  ...Overrides.Items,
  Card: Resources.resolved["Card.js"],
  CardActions: Resources.resolved["CardActions.js"],
  CardMetadata: Resources.resolved["CardMetadata.js"],
  CardTitle: Resources.resolved["CardTitle.js"],
  WidgetHeader: Resources.resolved["WidgetHeader.js"],
  WidgetHeaderWrapper: Resources.resolved["WidgetHeaderWrapper.js"],
  SingleCardComponent: Resources.resolved["SingleCardComponent.js"],
};

const EditForm = Resources.resolved["EditForm.js"];
const StatusComponent = Resources.resolved["StatusComponent.js"];
const {
  isStatusTrackingEnabled,
  generateLabel,
  getEditAndDeletePermissions,
  getDefaultStatusesForRole,
} = Resources.resolved["api.js"];
const { SnackbarProvider } = Resources.resolved["hooks.js"];
const { getEditDeletePermissions } = Resources.resolved["utils.js"];

const getItemSideActions = (defaultFn) => {
  const widget = React.useContext(WidgetContexts.Items);
  const {
    setIsEditModalOpen,
    setActiveItem,
    project,
    editDeletePermissions,
    user,
  } = widget.mainJsProps;
  const activeItemModel = widget.props.activeItem;
  const activeItem = activeItemModel.attributes;

  // Status based permissions
  const { canEdit } = getEditDeletePermissions(
    user,
    editDeletePermissions,
    activeItem,
  );

  const { useSnackbar } = Resources.resolved["hooks.js"];
  const { showSnackbar } = useSnackbar();

  const defaultActions = defaultFn();

  const itemsToHide = [
    "bookmark_outlined",
    "share",
    "mail_outlined",
    "print",
    "download",
  ];

  const actions = defaultActions.filter(
    (action) => action.icon && !itemsToHide.includes(action.icon),
  );

  const generateLabelAction = () => {
    if (!activeItem?.keywords?.company_name?.[0]) {
      return showSnackbar("Document has no company name.", "error");
    }
    generateLabel(activeItem.project_id, activeItem.id);
  };

  const onDownloadClick = () => {
    const wfiDocumentId = activeItem.keywords?.wfi_document_id?.[0];

    if (!wfiDocumentId) {
      showSnackbar(
        "Document has not been stored into WFI. Please try again later.",
        "error",
      );
      return;
    }

    window.open(
      `/studio/storage/wfi/download/${project.id}/${wfiDocumentId}`,
      "_blank",
    );
  };

  const onEditClick = () => {
    if (!canEdit) {
      return showSnackbar(
        "You do not have permission to perform this action",
        "error",
      );
    }
    setActiveItem(activeItemModel);
    setIsEditModalOpen(true);
  };

  actions.push(
    {
      icon: "download",
      name: "Download",
      onClick: onDownloadClick,
    },
    {
      icon: "edit",
      name: "Edit",
      onClick: onEditClick,
    },
    {
      icon: "assignment",
      name: "Generate Label",
      onClick: generateLabelAction,
    },
  );

  return actions;
};

export default (props) => {
  const [isEditModalOpen, setIsEditModalOpen] = React.useState(false);
  const [activeItem, setActiveItem] = React.useState(null);
  const [isStatusEnabled, setStatusEnabled] = React.useState(false);
  const [editDeletePermissions, setEditDeletePermissions] = React.useState({});
  const [itemsSelectedBulkAssign, setItemsSelectedBulkAssign] = React.useState({
    currentDocStatus: {
      code: "",
      header: "",
    },
    selectedIds: new Set(),
  });

  const { addFacetValue } = Hooks.useDrilldown(
    props.dashboardState,
    props.widgetModel,
  );

  const roleOCBC = props.user.get("user_information")?.role_ocbc || [];
  const config = props.widgetModel?.toJSON();
  const additionalWidgetConfig = {
    showTitle: config.showTitle,
    showExportOptions: config.showExportOptions,
    showViewToggler: config.showViewToggler,
    respondToToggle: config.respondToToggle,
    keywordsToExport: config.keywordsToExport,
    editFormFields: config.editFormFields,
    markCorrectPredictions: config.markCorrectPredictions,
    showDownloadIcon: config.showDownloadIcon,
    showEditIcon: config.showEditIcon,
    showCoverPageIcon: config.showCoverPageIcon,
    showDeleteIcon: config.showDeleteIcon,
    deleteIconVisibility: config.deleteIconVisibility,
    deleteIconVisibleRoles: config.deleteIconVisibleRoles,
  };

  React.useEffect(() => {
    if (props.dashboard.get("title") === "WIP") {
      getDefaultStatusesForRole(props.project.get("id"), roleOCBC[0]).then(
        (statuses) => {
          const query = statuses
            .map((statusCode) => `current_doc_status:${statusCode}`)
            .join(" OR ");
          addFacetValue(null, `(${query})`);
        },
      );
    }
  }, []);

  React.useEffect(() => {
    isStatusTrackingEnabled(props.project.id, roleOCBC).then((res) => {
      setStatusEnabled(res && props.model.get("isStatusVisible"));
    });
    getEditAndDeletePermissions(props.project.id, roleOCBC).then((res) => {
      setEditDeletePermissions(res);
    });
  }, []);

  // Reset the state of selected items for bulk assign whenever query changes
  React.useEffect(() => {
    const resetItemsSelectedBulkAssign = () => {
      setItemsSelectedBulkAssign({
        currentDocStatus: "",
        currentDocStatusHeader: "",
        selectedIds: new Set(),
      });
    };
    props.dashboardState.on("change:query", resetItemsSelectedBulkAssign);

    return () =>
      props.dashboardState.off("change:query", resetItemsSelectedBulkAssign);
  }, []);

  const itemDetailApiOverrides = {
    getItemSideActions: getItemSideActions,
    getSidepanelItems: (defaultFunction) => {
      const sidePanels = defaultFunction();
      isStatusEnabled &&
        sidePanels.push({
          title: "Document Status",
          value: "document_status",
          icon: "pending_actions",
          Component: () => <StatusComponent />,
        });
      return sidePanels;
    },
    getSidepanelWidth: () => 480,
  };

  const facetsCollectionObject = Factories.Collections.Facets.create(
    props.dashboardState,
    props.project,
    {
      facet_fields: props.model
        .get("keywordsToShowDocumentStatus")
        .map(({ facet }) => facet),
    },
  );
  const facetCollection = Hooks.useFacetsCollection(
    props,
    facetsCollectionObject,
  );

  return (
    <WidgetContexts.Items.Provider
      value={{
        setIsEditModalOpen,
        setActiveItem,
        itemsSelectedBulkAssign,
        setItemsSelectedBulkAssign,
        editDeletePermissions,
        dashboardTitle: props.dashboard.attributes.title,
        project: props.project.attributes,
        user: props.user.attributes,
        dashboardState: props.dashboardState,
        additionalWidgetConfig: additionalWidgetConfig,
        keywordsToShowDocumentStatus: facetCollection.props.indexed,
      }}
    >
      <SnackbarProvider>
        <Widgets.Items
          {...props}
          overrides={overrides}
          itemDetailApiOverrides={itemDetailApiOverrides}
        />
        <MUI.Modal
          open={isEditModalOpen}
          onClose={() => setIsEditModalOpen(false)}
        >
          <div>
            <EditForm
              item={activeItem?.attributes}
              activeItemModel={activeItem}
              handleClose={() => setIsEditModalOpen(false)}
            />
          </div>
        </MUI.Modal>
      </SnackbarProvider>
    </WidgetContexts.Items.Provider>
  );
};
