const updateDocument = async (itemId, wfiDocumentId, projectId, labels) => {
  const res = await fetch(`/studio/documents/${itemId}`, {
    method: "PUT",
    headers: {
      "content-type": "application/json",
    },
    body: JSON.stringify({
      project_id: projectId,
      wfi_document_id: wfiDocumentId,
      labels: labels,
    }),
  });

  if (!res.ok) throw new Error(`Failed to update document ${itemId}.`);

  return res;
};

const updateSquirroLabels = async (itemId, projectId, labels) => {
  const res = await fetch(`/studio/documents/${itemId}`, {
    method: "PATCH",
    headers: {
      "content-type": "application/json",
    },
    body: JSON.stringify({
      project_id: projectId,
      labels: labels,
    }),
  });

  if (!res.ok) throw new Error("Failed to update squirro labels.");

  return res;
};

const bulkAssignSquirroLabels = async (itemIds, projectId, labels) => {
  const res = await fetch(`/studio/documents/bulk`, {
    method: "PATCH",
    headers: {
      "content-type": "application/json",
    },
    body: JSON.stringify({
      ids: itemIds,
      project_id: projectId,
      labels: labels,
    }),
  });

  if (!res.ok) throw new Error("Failed to bulk assign Squirro labels.");

  return res;
};

const getStatuses = async (projectId, documentId) => {
  const res = await fetch(
    `/studio/document_status_tracking/projects/${projectId}/documents/${documentId}`,
  );

  if (!res.ok)
    throw new Error(`Failed to get status for document ${documentId}`);

  return res.json();
};

const getDefaultStatusesForRole = async (projectId, role) => {
  const res = await fetch(
    `/studio/status_permissions/${projectId}/default_statuses/${role}`,
  );

  if (!res.ok)
    throw new Error(`Failed to get status for document ${documentId}`);

  return res.json();
};

const postStatus = async (projectId, documentId, data) => {
  const res = await fetch(
    `/studio/document_status_tracking/projects/${projectId}/documents/${documentId}/status`,
    {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
      },
      body: JSON.stringify(data),
    },
  );

  if (!res.ok)
    throw new Error(`Failed to post status for document ${documentId}.`);

  return res.json();
};

const bulkAssignStatus = async (projectId, itemIds, statusCode, remarks) => {
  const reqBody = {
    document_ids: itemIds,
    status_code: statusCode,
  };
  remarks && (reqBody.remarks = remarks);

  const res = await fetch(
    `/studio/document_status_tracking/projects/${projectId}/documents-bulk-status`,
    {
      method: "POST",
      headers: {
        "content-type": "application/json",
      },
      body: JSON.stringify(reqBody),
    },
  );
  const data = await res.json();

  if (!res.ok) throw new Error(data.message || "Failed to bulk assign status.");

  return data;
};

const getPdf = (projectId, documentId, sort) => {
  const sortingParam = sort === "date_desc" ? "DESC" : "ASC";
  return window.open(
    `/studio/document_status_tracking/projects/${projectId}/documents/${documentId}/status/pdf?sort=${sortingParam}`,
  );
};

const isStatusTrackingEnabled = async (projectId, userRole) => {
  const res = await fetch(
    `/studio/document_status_tracking/projects/${projectId}/config`,
  );
  if (!res.ok) throw new Error("Failed to fetch status tracking config.");

  const data = await res.json();
  const allowedRoles = data.config.status_map.access_roles;

  let isEnabled = false;
  for (const role of userRole) {
    if (allowedRoles.includes(role)) {
      isEnabled = true;
      break;
    }
  }

  return isEnabled;
};

const generateLabel = (projectId, documentId) =>
  window.open(
    `/studio/document_status_tracking/projects/${projectId}/documents/${documentId}/label/pdf`,
  );

const getEditAndDeletePermissions = async (projectId, role) => {
  // Such cases will only occur for Workbench:CSSupportMaker&Checker
  if (role.length > 1) role = ["Workbench:CSSupportMaker&Checker"];

  const res = await fetch(
    `/studio/status_permissions/${projectId}/permissions/${role}`,
  );
  if (!res.ok) throw new Error("Failed to fetch get edit and permissions");

  return res.json();
};

export default {
  getStatuses,
  getDefaultStatusesForRole,
  postStatus,
  bulkAssignStatus,
  getPdf,
  isStatusTrackingEnabled,
  updateDocument,
  updateSquirroLabels,
  bulkAssignSquirroLabels,
  generateLabel,
  getEditAndDeletePermissions,
};
