export default (props) => {
  const [isBulkAssignFormOpen, setIsBulkAssignFormOpen] = React.useState(false);

  const widget = React.useContext(WidgetContexts.Items);
  const { data } = widget.props;
  const { view } = widget.customProps;
  const {
    dashboardState,
    dashboardTitle,
    project,
    user,
    additionalWidgetConfig,
    itemsSelectedBulkAssign,
    setItemsSelectedBulkAssign,
  } = widget.mainJsProps;

  const { useSnackbar } = Resources.resolved["hooks.js"];
  const { showSnackbar } = useSnackbar();

  const sortOrder = dashboardState.getSortOrder();
  const totalMessage = `${props.total} results`;

  const toggleView = (value) => {
    localStorage.setItem("view", value);
    widget.setCustomProps({ view: value });
  };

  const selectItemsInBulk = (maxItems = 30) => {
    const dataCapped = data.slice(0, maxItems);

    // Get status of the first document and compare it with the rest
    const docStatus = {
      code: dataCapped[0].keywords.current_doc_status?.[0],
      header: dataCapped[0].keywords.current_doc_status_header?.[0],
    };
    const hasSameStatus = dataCapped.every(
      (item) => item.keywords.current_doc_status?.[0] === docStatus.code,
    );

    if (!hasSameStatus) {
      return showSnackbar("Documents have different statuses.", "error");
    }

    setItemsSelectedBulkAssign({
      currentDocStatus: docStatus,
      selectedIds: new Set(dataCapped.map((item) => item.id)),
    });
  };

  // Card view by default
  React.useEffect(() => {
    widget.setCustomProps({
      view: localStorage.getItem("view") || "card",
    });
  }, []);

  return (
    <MUI.Box sx={{ display: "flex", flexDirection: "column", width: "100%" }}>
      <MUI.Box
        sx={{
          display: "flex",
          justifyContent: "space-between",
          alignItems: "center",
          width: "100%",
          py: 0.5,
          px: 2,
        }}
      >
        <MUI.Stack
          spacing={2}
          direction="row"
          sx={{
            alignItems: "center",
          }}
        >
          {additionalWidgetConfig.showTitle && props.title && (
            <MUI.Typography noWrap sx={{ flex: 1, pb: 1 }} variant="h3">
              {props.title}
            </MUI.Typography>
          )}

          {props.showSort && (
            <SortDropdownComponent
              currentSort={
                props.currentSort ? props.currentSort : sortItems[0].value
              }
              sortOrder={sortOrder}
              onSortChange={(value) => dashboardState.setSort(value, value)}
              sortItems={[
                {
                  value: "date",
                  title: "Upload Date",
                },
                {
                  value: "document_date",
                  title: "Document Date",
                },
              ]}
            />
          )}

          {props.showTotal && (
            <MUI.Typography sx={{ color: "text.secondary" }} variant="body2">
              {totalMessage}
            </MUI.Typography>
          )}

          {itemsSelectedBulkAssign.selectedIds.size > 0 && (
            <>
              <MUI.Tooltip title="Bulk Assign">
                <MUI.IconButton
                  size="small"
                  onClick={() => setIsBulkAssignFormOpen(true)}
                >
                  <MUI.Icon>edit</MUI.Icon>
                </MUI.IconButton>
              </MUI.Tooltip>

              <MUI.Typography sx={{ color: "text.secondary" }} variant="body2">
                ( {itemsSelectedBulkAssign.selectedIds.size} selected )
              </MUI.Typography>

              <MUI.Typography sx={{ color: "text.secondary" }} variant="body2">
                Status: {itemsSelectedBulkAssign.currentDocStatus.header}
              </MUI.Typography>

              <BulkAssignForm
                open={isBulkAssignFormOpen}
                closeForm={() => setIsBulkAssignFormOpen(false)}
              />
            </>
          )}
        </MUI.Stack>
        <MUI.Stack
          spacing={1}
          direction="row"
          sx={{
            alignItems: "center",
          }}
        >
          {additionalWidgetConfig.showExportOptions && (
            <ExportOptions
              keywordsToExport={additionalWidgetConfig.keywordsToExport}
              dashboardState={dashboardState}
              user={user}
              project={project}
            />
          )}

          {additionalWidgetConfig.showViewToggler && (
            <MUI.Box>
              <Overrides.Actions.ActionIconButton
                iconName="view_headline"
                isActive={widget.customProps?.view === "list"}
                isStoreControlled
                onClick={() => toggleView("list")}
              />
              <Overrides.Actions.ActionIconButton
                iconName="view_list"
                isActive={widget.customProps?.view === "card"}
                isStoreControlled
                onClick={() => toggleView("card")}
              />
            </MUI.Box>
          )}
        </MUI.Stack>
      </MUI.Box>

      {/* Header Row for List View */}
      {view === "list" && additionalWidgetConfig.respondToToggle && (
        <MUI.Box
          sx={{
            pr: 2,
            pl: dashboardTitle === "WIP" ? 0 : 2,
            py: 1.5,
            position: "relative",
            width: "100%",
            backgroundColor: "#ffffff",
            border: 1,
            borderColor: "divider",
            borderTopLeftRadius: "8px",
            borderTopRightRadius: "8px",
          }}
        >
          <MUI.Grid
            container
            spacing={1.5}
            sx={{
              alignItems: "center",
            }}
          >
            {dashboardTitle === "WIP" && (
              <>
                <MUI.Grid item xs={0.5}>
                  {itemsSelectedBulkAssign.selectedIds.size ? (
                    <MUI.Checkbox
                      sx={{ position: "absolute", top: 0, bottom: 0 }}
                      disableRipple
                      indeterminate={itemsSelectedBulkAssign.selectedIds.size}
                      onClick={() =>
                        setItemsSelectedBulkAssign({
                          currentDocStatus: {
                            code: "",
                            header: "",
                          },
                          selectedIds: new Set(),
                        })
                      }
                    />
                  ) : (
                    <MUI.Checkbox
                      sx={{ position: "absolute", top: 0, bottom: 0 }}
                      disableRipple
                      onClick={() => selectItemsInBulk()}
                    />
                  )}
                </MUI.Grid>
              </>
            )}
            <MUI.Grid item xs={1}>
              <ListViewHeading>Current Status</ListViewHeading>
            </MUI.Grid>
            <MUI.Grid item xs={1}>
              <ListViewHeading>Title</ListViewHeading>
            </MUI.Grid>
            <MUI.Grid item xs={dashboardTitle === "WIP" ? 1.3 : 1.8}>
              <ListViewHeading>Company Name</ListViewHeading>
            </MUI.Grid>
            <MUI.Grid item xs={1}>
              <ListViewHeading>CSO Name</ListViewHeading>
            </MUI.Grid>
            <MUI.Grid item xs={1.1}>
              <ListViewHeading>Document Type</ListViewHeading>
            </MUI.Grid>
            <MUI.Grid item xs={1}>
              <ListViewHeading>Document Date</ListViewHeading>
            </MUI.Grid>
            <MUI.Grid item xs={1}>
              <ListViewHeading>References</ListViewHeading>
            </MUI.Grid>
            <MUI.Grid item xs={1}>
              <ListViewHeading>RM Name</ListViewHeading>
            </MUI.Grid>
            <MUI.Grid item xs={1.1}>
              <ListViewHeading>Upload Date</ListViewHeading>
            </MUI.Grid>
            <MUI.Grid item xs={1}>
              <ListViewHeading>Upload By</ListViewHeading>
            </MUI.Grid>
            <MUI.Grid item xs={1}>
              <ListViewHeading># of Pages</ListViewHeading>
            </MUI.Grid>
          </MUI.Grid>
        </MUI.Box>
      )}
    </MUI.Box>
  );
};

const ExportOptions = ({ keywordsToExport, dashboardState, user, project }) => {
  const [filters, setFilters] = React.useState({
    rm_name: [],
    wfi_company_segment: [],
    wfi_company_team_name: [],
    current_doc_status_header: [],
  });

  const [dates, setDates] = React.useState({
    createdAfter: "",
    createdBefore: "",
  });
  const [dateErrorText, setDateErrorText] = React.useState("");

  const { AutocompleteFromAPI } = Resources.resolved["Inputs.js"];
  const { getCurrentQuery } = Resources.resolved["utils.js"];

  const isUserRm = user.user_information?.role_ocbc?.[0] === "Workbench:RM";
  const segmentMapping = { R: "RE", W: "WCM" };

  const [menuAnchorEl, setMenuAnchorEl] = React.useState(null);
  const menuOpen = Boolean(menuAnchorEl);
  const [modalOpen, setModalOpen] = React.useState(false);

  const handleMenuOpen = (event) => {
    setMenuAnchorEl(event.currentTarget);
  };

  const handleMenuClose = () => {
    setMenuAnchorEl(null);
  };

  const handleModalOpen = () => {
    setModalOpen(true);
  };

  const handleModalClose = () => {
    setModalOpen(false);
    resetInputs();
  };

  const resetInputs = () => {
    setFilters({
      rm_name: [],
      wfi_company_segment: [],
      wfi_company_team_name: [],
      current_doc_status_header: [],
    });
    setDates({ createdAfter: "", createdBefore: "" });
    setDateErrorText("");
  };

  const saveToExcel = () => {
    const query = getCurrentQuery(App.globalSearch, dashboardState);

    const parameters = {
      project_id: project.id,
      query: query ? query : "",
      keywords: JSON.stringify(keywordsToExport),
      dashboard: dashboardState.options.dashboard.getTitle() || "Squirro",
    };

    const url = "/studio/documents/save_to_excel?" + $.param(parameters);
    window.open(url, "_blank");
  };

  const generateReport = () => {
    if (!dates.createdBefore && !dates.createdAfter) {
      return setDateErrorText(
        "Please select at least one of Created After or Created Before.",
      );
    }

    let query = Object.entries(filters)
      .filter(([__, values]) => !_.isEmpty(values))
      .map(
        ([label, values]) =>
          "(" + values.map((val) => `${label}:"${val}"`).join(" OR ") + ")", // OR for same label's values
      )
      .join(" AND "); // AND to join different labels

    query = `${query} ${dashboardState.getDashboardQuery()}`;

    const parameters = {
      project_id: project.id,
      query: query,
      created_before: dates.createdBefore
        ? dates.createdBefore.utc().format("YYYY-MM-DDTHH:mm:ss")
        : "",
      created_after: dates.createdAfter
        ? dates.createdAfter.utc().format("YYYY-MM-DDTHH:mm:ss")
        : "",
    };

    const url = "/studio/documents/generate_report?" + $.param(parameters);
    window.open(url, "_blank");
  };

  return (
    <>
      <MUI.Button
        sx={{
          color: "grey",
          "&:hover": {
            backgroundColor: theme.palette.hover,
          },
        }}
        variant="text"
        onClick={handleMenuOpen}
        endIcon={<MUI.Icon>keyboard_arrow_down</MUI.Icon>}
      >
        Export
      </MUI.Button>
      <MUI.Menu
        anchorEl={menuAnchorEl}
        open={menuOpen}
        onClose={handleMenuClose}
      >
        <MUI.MenuItem onClick={saveToExcel}>Save Current Results</MUI.MenuItem>
        <MUI.MenuItem
          onClick={() => {
            handleMenuClose();
            handleModalOpen();
          }}
        >
          Generate Report
        </MUI.MenuItem>
      </MUI.Menu>
      <MUI.Modal open={modalOpen} onClose={handleModalClose}>
        <MUI.Box
          sx={{
            position: "absolute",
            top: "50%",
            left: "50%",
            maxHeight: "90%",
            overflowY: "auto",
            transform: "translate(-50%, -50%)",
            margin: "auto",
            backgroundColor: "#ffffff",
            boxShadow: 24,
            borderRadius: "8px",
            p: 3,
            width: {
              lg: "50%",
              xs: "80%",
            },
          }}
        >
          <MUI.Stack spacing={2}>
            <MUI.Typography sx={{ fontWeight: "bold" }} variant="h6">
              Generate Report
            </MUI.Typography>
            <MUI.Stack spacing={2} direction="row">
              <MUI.FormControl fullWidth>
                <MUI.Autocomplete
                  multiple
                  disabled={isUserRm}
                  value={
                    (isUserRm && [user.user_information?.rmcode?.[0]?.[0]]) || // First Letter
                    filters.wfi_company_segment
                  }
                  onChange={(e, segments) => {
                    setFilters({
                      ...filters,
                      wfi_company_segment: segments.map(
                        (item) => item.value ?? item,
                      ),
                    });
                  }}
                  options={Object.entries(segmentMapping).map(
                    ([value, label]) => ({
                      value,
                      label,
                    }),
                  )}
                  isOptionEqualToValue={(option, value) =>
                    option.value == value
                  }
                  renderInput={(params) => (
                    <MUI.TextField {...params} label="Segment" />
                  )}
                  renderTags={(tags, getTagProps) =>
                    tags.map((value, index) => (
                      <MUI.Chip
                        {...getTagProps({ index })}
                        label={segmentMapping[value]}
                      />
                    ))
                  }
                />
              </MUI.FormControl>
              <MUI.FormControl fullWidth>
                <AutocompleteFromAPI
                  multiple
                  label="RM Name"
                  value={filters.rm_name}
                  onChange={(e, rmNames) =>
                    setFilters({
                      ...filters,
                      rm_name: rmNames,
                    })
                  }
                  autoCompleteAPIUrl={`/studio/autocomplete_api/rm_names/${project.id}?q=`}
                  fetchOptionsOnInputChange
                />
              </MUI.FormControl>
            </MUI.Stack>

            <MUIXDatePicker.LocalizationProvider
              dateAdapter={MUIXAdapterMoment}
            >
              <MUI.Stack direction="row" spacing={2}>
                <MUI.FormControl fullWidth>
                  <MUIXDatePicker.DatePicker
                    label="Created After"
                    onChange={(date) => {
                      setDateErrorText("");
                      setDates({
                        ...dates,
                        createdAfter: date.clone(),
                      });
                    }}
                    format="DD/MM/YYYY"
                    disableFuture={true}
                    slotProps={{
                      textField: {
                        error: !!dateErrorText,
                        helperText: dateErrorText,
                      },
                    }}
                  />
                </MUI.FormControl>
                <MUI.FormControl fullWidth>
                  <MUIXDatePicker.DatePicker
                    label="Created Before"
                    onChange={(date) => {
                      setDateErrorText("");
                      setDates({
                        ...dates,
                        createdBefore: date.clone(),
                      });
                    }}
                    format="DD/MM/YYYY"
                    disableFuture={true}
                    slotProps={{
                      textField: {
                        error: !!dateErrorText,
                      },
                    }}
                  />
                </MUI.FormControl>
              </MUI.Stack>
            </MUIXDatePicker.LocalizationProvider>

            <MUI.Stack direction="row" spacing={2}>
              <MUI.FormControl fullWidth>
                <AutocompleteFromAPI
                  multiple
                  label="Current Status"
                  value={filters.current_doc_status_header}
                  onChange={(e, statuses) =>
                    setFilters({
                      ...filters,
                      current_doc_status_header: statuses,
                    })
                  }
                  autoCompleteAPIUrl={`/studio/document_status_tracking/projects/${project.id}/config`}
                  getOptions={(data) =>
                    data.config.status_map.statuses
                      .map(({ header }) => header)
                      .sort()
                  }
                />
              </MUI.FormControl>
              <MUI.FormControl fullWidth>
                <AutocompleteFromAPI
                  multiple
                  label="Team"
                  value={filters.wfi_company_team_name}
                  onChange={(e, teams) =>
                    setFilters({ ...filters, wfi_company_team_name: teams })
                  }
                  autoCompleteAPIUrl={
                    `/v0/items/query?query=&project_id=${project.id}` +
                    `&count=0&aggregations=${encodeURIComponent(
                      '{"wfi_company_team_name":{"size":1000}}',
                    )}`
                  }
                  getOptions={(data) =>
                    data.aggregations.wfi_company_team_name.wfi_company_team_name.values.map(
                      (item) => item.key,
                    )
                  }
                />
              </MUI.FormControl>
            </MUI.Stack>

            <MUI.Stack
              direction="row"
              spacing={2}
              sx={{ alignSelf: "flex-end" }}
            >
              <MUI.Button onClick={handleModalClose}>Cancel</MUI.Button>
              <MUI.Button onClick={generateReport} variant="contained">
                Export
              </MUI.Button>
            </MUI.Stack>
          </MUI.Stack>
        </MUI.Box>
      </MUI.Modal>
    </>
  );
};

const SortSelect = styled(MUI.Select)(({ theme, size }) => ({
  border: "none",
  lineHeight: "14px",
  textDecoration: "none",

  ".MuiSelect-select": {
    ...theme.typography.body2,
    color: theme.palette.text.secondary,
    fontSize: size === "small" ? 12 : 14,
  },

  ".MuiOutlinedInput-input": {
    paddingTop: 0,
    paddingBottom: 0,
    paddingLeft: 0,
  },

  ".MuiOutlinedInput-notchedOutline": {
    border: "none !important",
  },
}));

const StyledMenuItem = styled(MUI.MenuItem)(({ theme }) => ({
  "&.Mui-selected": {
    backgroundColor: theme.palette.custom.white + " !important",

    "&:hover": {
      backgroundColor: theme.palette.background.lightGray + " !important",
    },
  },
}));

const SortDropdownComponent = ({
  sortItems,
  onSortChange,
  sortOrder,
  variant,
  label,
  disabled,
  size,
}) => {
  const [sortDisplayText, setSortDisplayText] = React.useState("");

  const handleChange = (value) => {
    onSortChange(value);
  };

  return (
    <MUI.FormControl variant={variant || "outlined"} size="small">
      {label && <MUI.InputLabel>{label}</MUI.InputLabel>}

      <SortSelect
        disabled={disabled}
        label={label}
        onChange={(e) => setSortDisplayText(e.target.value)}
        size={size}
        displayEmpty
        renderValue={sortDisplayText !== "" ? undefined : () => "Sort By"}
        value={sortDisplayText}
        startAdornment={
          sortOrder && (
            <MUI.Icon sx={{ mr: 1 }} fontSize="small">
              {sortOrder === "asc" ? "arrow_upward" : "arrow_downward"}
            </MUI.Icon>
          )
        }
      >
        {sortItems?.map((item, index) => (
          <StyledMenuItem
            onClick={() => {
              handleChange(item.value);
            }}
            key={index}
            value={item.value}
          >
            {item.title}
          </StyledMenuItem>
        ))}
      </SortSelect>
    </MUI.FormControl>
  );
};

const ListViewHeading = styled(MUI.Typography)({
  fontWeight: "bold",
});

ListViewHeading.defaultProps = {
  color: "textPrimary",
  variant: "body2",
  component: "p",
  noWrap: true,
};

const BulkAssignForm = ({ open, closeForm }) => {
  const [loading, setLoading] = React.useState(false);
  const [csoName, setCsoName] = React.useState("");
  const [remarks, setRemarks] = React.useState("");
  const [possibleStatuses, setPossibleStatuses] = React.useState([]);
  const [statusCode, setStatusCode] = React.useState("");
  const selectedStatus = possibleStatuses.find(
    (possibleStatus) => possibleStatus.code === statusCode,
  );

  const widget = React.useContext(WidgetContexts.Items);
  const { project, itemsSelectedBulkAssign, setItemsSelectedBulkAssign } =
    widget.mainJsProps;

  const { getStatuses, bulkAssignSquirroLabels, bulkAssignStatus } =
    Resources.resolved["api.js"];
  const { AutocompleteFromAPI } = Resources.resolved["Inputs.js"];
  const { useSnackbar } = Resources.resolved["hooks.js"];
  const { showSnackbar } = useSnackbar();

  React.useEffect(() => {
    if (open) {
      getStatuses(
        project.id,
        Array.from(itemsSelectedBulkAssign.selectedIds)[0],
      ).then((data) => setPossibleStatuses(data.document.next_statuses));
    } else {
      setPossibleStatuses([]);
    }
  }, [open]);

  const handleSubmit = async () => {
    setLoading(true);

    try {
      if (csoName) {
        await bulkAssignSquirroLabels(
          Array.from(itemsSelectedBulkAssign.selectedIds),
          project.id,
          { cso_name: [csoName] },
        );
      }

      if (statusCode) {
        await bulkAssignStatus(
          project.id,
          Array.from(itemsSelectedBulkAssign.selectedIds),
          statusCode,
          remarks,
        );
      }

      showSnackbar("Documents have been updated.");

      // Update item context to show the updated keywords
      Array.from(itemsSelectedBulkAssign.selectedIds).forEach((itemId) => {
        const itemModel = widget.props?.collection?.get(itemId);
        const updatedKeywords = {
          ...itemModel.get("keywords"),
        };

        if (csoName) {
          updatedKeywords.cso_name = [csoName];
        }

        if (statusCode) {
          updatedKeywords.current_doc_status = [selectedStatus.code];
          updatedKeywords.current_doc_status_header = [selectedStatus.header];
        }

        itemModel.set("keywords", updatedKeywords);
      });
      widget.props.collection.trigger("updated:model");

      // Update the selected items state
      if (statusCode) {
        setItemsSelectedBulkAssign({
          ...itemsSelectedBulkAssign,
          currentDocStatus: {
            code: selectedStatus.code,
            header: selectedStatus.header,
          },
        });
      }
    } catch (error) {
      return showSnackbar(error.message, "error");
    } finally {
      setLoading(false);
      resetInputs();
      closeForm();
    }
  };

  const resetInputs = () => {
    setCsoName("");
    setRemarks("");
    setStatusCode("");
  };

  return (
    <MUI.Modal
      open={open}
      onClose={() => {
        resetInputs();
        closeForm();
      }}
    >
      <MUI.Box
        sx={{
          position: "absolute",
          top: "50%",
          left: "50%",
          maxHeight: "90%",
          overflowY: "auto",
          transform: "translate(-50%, -50%)",
          margin: "auto",
          p: "24px",
          backgroundColor: "#ffffff",
          boxShadow: 24,
          borderRadius: "8px",
          width: {
            lg: "50%",
            xs: "80%",
          },
        }}
      >
        <MUI.Stack spacing={2}>
          <MUI.Typography
            sx={{ mt: "0 !important", fontWeight: "bold" }}
            variant="body1"
            component="h4"
          >
            Bulk Assign
          </MUI.Typography>
          <AutocompleteFromAPI
            label="CSO Name"
            onChange={(e, val) => setCsoName(val)}
            autoCompleteAPIUrl={`/studio/autocomplete_api/cso_names/${project.id}?q=`}
            fetchOptionsOnInputChange
          />
          <MUI.Tooltip
            arrow
            placement="top"
            title={
              possibleStatuses.length === 0
                ? "You don't have rights to submit a new status"
                : undefined
            }
          >
            <MUI.FormControl disabled={possibleStatuses.length === 0} fullWidth>
              <MUI.InputLabel id="segment">Status</MUI.InputLabel>
              <MUI.Select
                fullWidth
                disabled={possibleStatuses.length === 0}
                placeholder="Select Document Status"
                value={statusCode}
                onChange={(event) => setStatusCode(event.target.value)}
              >
                {possibleStatuses.map(({ header, code }) => (
                  <MUI.MenuItem value={code}>{header}</MUI.MenuItem>
                ))}
              </MUI.Select>
            </MUI.FormControl>
          </MUI.Tooltip>

          {selectedStatus && selectedStatus?.remarks !== "forbidden" && (
            <MUI.TextField
              multiline
              variant="outlined"
              label={`Remarks (${selectedStatus.remarks})`}
              required={selectedStatus.remarks === "mandatory"}
              error={
                selectedStatus.remarks === "mandatory" && remarks.length === 0
              }
              inputProps={{ maxLength: 250 }}
              helperText={`${remarks.length}/250`}
              value={remarks}
              onChange={(event) => setRemarks(event.target.value)}
            />
          )}
          <MUI.Stack direction="row" justifyContent="flex-end" spacing={2}>
            <MUI.Button
              onClick={() => {
                resetInputs();
                closeForm();
              }}
            >
              Cancel
            </MUI.Button>
            <MUI.Button
              disabled={!csoName && !statusCode}
              variant="contained"
              onClick={handleSubmit}
            >
              Save
            </MUI.Button>
          </MUI.Stack>
          {loading && (
            <MUI.Backdrop
              sx={{ backgroundColor: "rgba(255, 255, 255, 0.5)" }}
              open={true}
            >
              <MUI.CircularProgress />
            </MUI.Backdrop>
          )}
        </MUI.Stack>
      </MUI.Box>
    </MUI.Modal>
  );
};
