const {
  Stack,
  Box,
  Typography,
  Card,
  CardHeader,
  CardContent,
  CardActions,
  Chip,
  IconButton,
  Icon,
  Button,
  TextField,
  Select,
  InputLabel,
  MenuItem,
  CircularProgress,
  Alert,
  AlertTitle,
  Tooltip,
} = MUI;

export default ({ data, sorting, postStatus }) => {
  const { document } = data;

  const indexedStatuses = document.status_trail
    .sort((a, b) => new Date(a.timestamp) - new Date(b.timestamp))
    .map((status, index) => ({ ...status, index: index + 1 }))
    .reverse();

  const reversed = sorting === "date_asc";

  return (
    <Stack
      alignItems="start"
      direction={reversed ? "column-reverse" : "column"}
      sx={{ width: "100%" }}
    >
      <SubmitForm
        reversed={reversed}
        possibleStatuses={document.next_statuses}
        postStatus={postStatus}
      />
      {indexedStatuses.map((status, index) => (
        <StatusCard
          {...status}
          reversed={reversed}
          key={index}
          length={indexedStatuses.length}
        />
      ))}
    </Stack>
  );
};

const SubmitForm = ({ reversed, possibleStatuses, postStatus }) => {
  const context = React.useContext(WidgetContexts.Items);

  const [isExpanded, setExpanded] = React.useState(false);
  const [isLoading, setLoading] = React.useState(false);
  const [error, setError] = React.useState();
  const [statusCode, setStatusCode] = React.useState();
  const [remarks, setRemarks] = React.useState("");

  const disabled = possibleStatuses.length === 0;
  const selectedStatus = possibleStatuses.find((el) => el.code === statusCode);
  const userRole = context.mainJsProps.user?.user_information?.role;

  const submitForm = async () => {
    if (selectedStatus?.remarks === "mandatory" && remarks.length === 0) {
      return;
    }

    setExpanded(false);
    setLoading(true);

    try {
      await postStatus({
        status_code: statusCode,
        remarks,
      });
      setLoading(false);
    } catch (error) {
      setError(error);
      setLoading(false);
    }
  };

  const discardForm = () => {
    setStatusCode();
    setRemarks("");
    setExpanded(false);
  };

  return (
    <MapEntry>
      <TimelineComponent
        dashed
        first
        index="+"
        reversed={reversed}
        error={!!error}
      />
      {!isExpanded && !isLoading && !error && (
        <Tooltip
          arrow
          placement="top"
          title={
            disabled
              ? "You don't have rights to submit a new status"
              : undefined
          }
        >
          <div>
            <Button
              variant="outlined"
              color="secondary"
              disabled={disabled}
              sx={{ my: "6px !important" }}
              startIcon={<Icon>add</Icon>}
              onClick={() => setExpanded(true)}
            >
              Submit New Status
            </Button>
          </div>
        </Tooltip>
      )}
      {isExpanded && (
        <Card sx={{ display: "block", width: "100%" }}>
          <CardHeader
            disableTypography
            sx={{
              backgroundColor: "#FAFAFA",
              borderBottom: isExpanded ? "1px solid #E0E0E0" : undefined,
              py: 1,
            }}
            title={
              <Typography
                color="text.secondary"
                fontWeight={500}
                sx={{ mr: "auto" }}
              >
                Submit New Status
              </Typography>
            }
          />
          <CardContent sx={{ p: 2 }}>
            <Stack spacing={1}>
              <Box>
                <InputLabel variant="standard">Status</InputLabel>
                <Select
                  fullWidth
                  variant="standard"
                  placeholder="Select Document Status"
                  value={statusCode}
                  onChange={(event) => setStatusCode(event.target.value)}
                >
                  {possibleStatuses.map(({ header, code }) => (
                    <MenuItem value={code}>{header}</MenuItem>
                  ))}
                </Select>
              </Box>
              <TextField
                disabled
                variant="standard"
                label="Reporter"
                value={`${context.mainJsProps.user.fullName} ${
                  userRole ? `(${userRole})` : ""
                }`}
              />
              {selectedStatus && selectedStatus?.remarks !== "forbidden" && (
                <TextField
                  multiline
                  variant="standard"
                  label={`Remarks (${selectedStatus.remarks})`}
                  required={selectedStatus.remarks === "mandatory"}
                  error={
                    selectedStatus.remarks === "mandatory" &&
                    remarks.length === 0
                  }
                  inputProps={{ maxLength: 250 }}
                  helperText={`${remarks.length}/250`}
                  value={remarks}
                  onChange={(event) => setRemarks(event.target.value)}
                />
              )}
            </Stack>
          </CardContent>
          <CardActions>
            <Button
              disableElevation
              variant="contained"
              color="secondary"
              disabled={selectedStatus === undefined}
              onClick={submitForm}
            >
              Submit Status
            </Button>
            <Button variant="outlined" color="secondary" onClick={discardForm}>
              Discard
            </Button>
          </CardActions>
        </Card>
      )}
      {isLoading && (
        <Card sx={{ display: "block", width: "100%" }}>
          <CardHeader
            disableTypography
            sx={{
              backgroundColor: "#FAFAFA",
              borderBottom: isExpanded ? "1px solid #E0E0E0" : undefined,
              py: 1,
            }}
            title={
              <Stack direction="row" spacing={1} alignItems="center">
                <CircularProgress size={16} />
                <Typography
                  color="text.secondary"
                  fontWeight={500}
                  sx={{ mr: "auto" }}
                >
                  Submiting
                </Typography>
              </Stack>
            }
          />
        </Card>
      )}
      {!!error && (
        <Alert severity="error" sx={{ width: "100%" }}>
          <AlertTitle>Unexpected Error</AlertTitle>
          Error with status code {error.status} has occured. Please, refresh the
          page and try again.
        </Alert>
      )}
    </MapEntry>
  );
};

const MapEntry = (props) => (
  <Stack
    direction="row"
    spacing={1}
    sx={{ width: "100%", position: "relative" }}
    {...props}
  />
);

const TimelineComponent = ({ index, reversed, first, last, dashed, error }) => (
  <>
    <Circle reversed={reversed} first={first} error={error} value={index}>
      {index}
    </Circle>
    <Line reversed={reversed} dashed={dashed} first={first} last={last} />
  </>
);

const Circle = styled("div")(({ theme, reversed, first, error }) => ({
  height: 20,
  width: 20,
  marginTop: `${first ? 14 : reversed ? 14 : 22}px !important`,
  backgroundColor: error
    ? theme.palette.error.main
    : theme.palette.secondary.main,
  color: "#FFF",
  fontWeight: 500,
  borderRadius: "50%",
  fontSize: 12,
  textAlign: "center",
  zIndex: 1,
}));

const Line = styled("div")(({ theme, dashed, first, last, reversed }) => ({
  position: "absolute",
  left: 1,
  top: 34,
  borderColor: theme.palette.secondary.main,
  borderRightStyle: dashed ? "dashed" : "solid",
  borderRightWidth: 2,
  height: (last && !reversed) || (first && reversed) ? 0 : "100%",
  zIndex: 0,
}));

const StatusCard = ({
  header,
  timestamp,
  description,
  user,
  remarks,
  index,
  reversed,
  length,
}) => {
  const { formatTimestamp } = Resources.resolved["utils.js"];

  const [isExpanded, setExpanded] = React.useState(index === length);
  const userRole = user.role?.[0];

  return (
    <MapEntry>
      <TimelineComponent
        index={index}
        reversed={reversed}
        last={index === 1}
        dashed={index === length && reversed}
      />
      <Card
        sx={{
          display: "block",
          width: "100%",
          [reversed ? "mb" : "mt"]: "8px !important",
        }}
      >
        <CardHeader
          disableTypography
          sx={{
            backgroundColor: "#FAFAFA",
            borderBottom: isExpanded ? "1px solid #E0E0E0" : undefined,
            py: 1,
          }}
          title={
            <Stack
              direction="row"
              spacing={1}
              alignItems="center"
              justifyContent="space-between"
            >
              <Typography
                color="text.secondary"
                fontWeight={500}
                sx={{ mr: "auto" }}
              >
                {header}
              </Typography>
              <Chip label={formatTimestamp(timestamp)} />
            </Stack>
          }
          action={
            <IconButton
              className={`expand-button ${
                isExpanded ? "expand-button-expanded" : "expand-button-hidden"
              }`}
              sx={{ ml: 1 }}
              onClick={() => setExpanded(!isExpanded)}
            >
              <Icon>{isExpanded ? "arrow_drop_up" : "arrow_drop_down"}</Icon>
            </IconButton>
          }
        />
        {isExpanded && (
          <CardContent sx={{ p: 2 }}>
            <Stack spacing={1}>
              {description && <Typography>{description}</Typography>}
              {user?.name && (
                <Typography color="text.secondary">
                  Reporter: {user.name} {userRole ? `(${userRole})` : ""}
                </Typography>
              )}
              {remarks && (
                <Typography color="text.secondary">
                  Remarks: {remarks}
                </Typography>
              )}
            </Stack>
          </CardContent>
        )}
      </Card>
    </MapEntry>
  );
};
