const {
  Stack,
  Typography,
  Select,
  MenuItem,
  CircularProgress,
  Alert,
  AlertTitle,
  Tooltip,
  IconButton,
  Icon,
} = MUI;

export default () => {
  const context = React.useContext(WidgetContexts.Items);
  const activeItem = context.props.activeItem;

  const { getStatuses, postStatus, getPdf } = Resources.resolved["api.js"];

  const [sorting, setSorting] = React.useState("date_desc");
  const [statusData, setStatusData] = React.useState({
    loading: true,
    error: false,
    data: {},
  });

  const itemData = React.useContext(WidgetContexts.Items);
  let documentTitle = itemData.props.activeItem.get("title");
  documentTitle = documentTitle.replace(/\<\/?squirro:highlight\>/g, "");
  const documentKeywords = itemData.props.activeItem.get("keywords");

  const keywordsToShow = Object.entries(
    itemData.mainJsProps.keywordsToShowDocumentStatus,
  )
    .map(([key, value]) =>
      value !== undefined ? { key, name: value.display_name } : null,
    )
    .filter((entry) => entry !== null);

  const documentLabels = keywordsToShow.map(({ key, name }) => ({
    name,
    key,
    value: documentKeywords?.[key]?.join(", "),
  }));

  React.useEffect(() => {
    const loadStatuses = () =>
      getStatuses(activeItem.project_id, activeItem.id)
        .then((data) => setStatusData({ loading: false, error: false, data }))
        .catch((error) => setStatusData({ loading: false, error, data: {} }));

    loadStatuses();
  }, []);

  const newStatus = async (data) => {
    return postStatus(activeItem.project_id, activeItem.id, data).then((data) =>
      setStatusData({ loading: false, error: false, data }),
    );
  };

  const downloadPdf = statusData.error
    ? undefined
    : () => {
        getPdf(activeItem.project_id, activeItem.id, sorting);
      };

  return (
    <Stack sx={{ height: "100%" }}>
      <LabelsTable title={documentTitle} labels={documentLabels} />
      <StatusHeader
        sorting={sorting}
        setSorting={setSorting}
        options={sortingOptions}
        downloadPdf={downloadPdf}
      />
      <StatusMapContainer
        statusData={statusData}
        sorting={sorting}
        postStatus={newStatus}
      />
    </Stack>
  );
};

const LabelsTable = ({ title, labels }) => (
  <Stack spacing={1} sx={{ p: 2 }}>
    <Typography fontSize={16} fontWeight={500}>
      {title}
    </Typography>
    {labels.map(
      (label) =>
        label?.name && (
          <LabelRow
            name={label.name}
            value={
              label.key === "document_date"
                ? moment(label.value).format("DD/MM/YYYY")
                : label.value
            }
          />
        ),
    )}
  </Stack>
);

const LabelRow = ({ name, value }) => (
  <Stack direction="row" spacing={1}>
    <Typography color="text.secondary">{name}:</Typography>
    <Typography>{value}</Typography>
  </Stack>
);

const StatusHeader = ({ sorting, setSorting, options, downloadPdf }) => (
  <Stack
    direction="row"
    justifyContent="space-between"
    alignItems="center"
    sx={{
      px: 2,
      py: 1.5,
      borderTop: "1px solid #E0E0E0",
      borderBottom: "1px solid #E0E0E0",
    }}
  >
    <Typography color="text.secondary" fontWeight={500}>
      Document Status
    </Typography>
    <Stack direction="row" alignItems="center" spacing={1}>
      <Typography color="text.secondary">Sort by</Typography>
      <SortingDropdown
        className="sorting-dorpdown"
        variant="filled"
        value={sorting}
        onChange={(event) => setSorting(event.target.value)}
      >
        {options.map((option) => (
          <MenuItem value={option.value}>{option.label}</MenuItem>
        ))}
      </SortingDropdown>
      <Tooltip arrow placement="top" title="Download as PDF">
        <IconButton disabled={downloadPdf === undefined} onClick={downloadPdf}>
          <Icon>download</Icon>
        </IconButton>
      </Tooltip>
    </Stack>
  </Stack>
);

const sortingOptions = [
  { label: "Date - Descending", value: "date_desc" },
  { label: "Date - Ascending", value: "date_asc" },
];

const SortingDropdown = styled(Select)(() => ({
  borderBottomLeftRadius: 8,
  borderBottomRightRadius: 8,

  ".MuiInputBase-input": {
    padding: "7px 12px",
  },

  "&:before": {
    borderBottom: "none !important",
  },

  "&:after": {
    borderBottom: "none !important",
  },
}));

const StatusMapContainer = ({ statusData, sorting, postStatus }) => {
  const StatusMap = Resources.resolved["StatusMap.js"];

  const { loading, error, data } = statusData;

  return (
    <Stack
      alignItems="center"
      sx={{
        p: 2,
        backgroundColor: "#FAFAFA",
        overflowY: "auto",
        flex: "1 1 auto",
      }}
    >
      {loading && <CircularProgress />}
      {error &&
        (error.status === 404 ? (
          <Alert severity="warning">
            <AlertTitle>No status tracking found</AlertTitle>
            This document doesn't have any statuses assigned to it.
          </Alert>
        ) : (
          <Alert severity="error">
            <AlertTitle>Error</AlertTitle>
            Something went wrong while fetching document status.
          </Alert>
        ))}
      {!loading && !error && (
        <StatusMap data={data} sorting={sorting} postStatus={postStatus} />
      )}
    </Stack>
  );
};
