export default (props) => {
  const [keywords, setKeywords] = React.useState(props.keywords || {});

  const widget = React.useContext(WidgetContexts.Items);
  const itemModel = widget.props.collection.get(props.id);
  const { additionalWidgetConfig } = widget.mainJsProps;
  const { view } = widget.customProps;

  const keywordsToDisplay = widget.props.config.keywordsToDisplay.map(
    (keyword) => keyword.facet,
  );

  if (itemModel) {
    itemModel.on("change", () => {
      const updatedKeywords = itemModel.get("keywords");
      const updatedKeywordsToDisplay = {};

      Object.entries(updatedKeywords).forEach(([label, value]) => {
        if (keywordsToDisplay.includes(label)) {
          updatedKeywordsToDisplay[label] = value;
        }
      });

      setKeywords(updatedKeywordsToDisplay);
    });
  }

  const cardModes = {
    isHorizontalCard: false,
    isHorizontalList: false,
    isVerticalCard: false,
    isVerticalList: true,
  };

  const additionalListViewProps = {
    showThumbnail: false,
    showMetadata: props.showMetadata || false,
    showTagsOnCard: false,
    showAbstract: false,
  };

  React.useEffect(() => {
    setKeywords(props.keywords || {});
  }, [props.keywords]);

  return (
    <>
      {additionalWidgetConfig.respondToToggle &&
      view === "list" &&
      /* We don't want to use custom list view for reference view. In order to check for
      the `isReferenceView` condition, we can only leverage showMetadata because it will
      always be set to false if it's in a reference view, and there are no other config
      exposed to custom widgets.

      https://github.com/squirro/squirro-integration/blob/
      2f37704d768c0be7a107a4348f983e73e0c82ed5/frontend/squirro/integration/frontend/userapp/
      static/js/require/widgets/Items/ItemDetail/components/ReferenceItemParent.tsx#L180

      This also means that showMetadata config on the dashboards have to be set to true
      for the custom list view to work.*/
      additionalListViewProps.showMetadata ? (
        <SingleCardListView
          {...props}
          {...additionalListViewProps}
          cardModes={cardModes}
          keywords={keywords}
        />
      ) : (
        <Overrides.Items.SingleCardComponent {...props} keywords={keywords} />
      )}
    </>
  );
};

const SingleCardListView = (props) => {
  const item = React.useContext(ModelContexts.Items);
  const widget = React.useContext(WidgetContexts.Items);
  const {
    dashboardTitle,
    itemsSelectedBulkAssign,
    setItemsSelectedBulkAssign,
  } = widget.mainJsProps;

  const [showCardActions, setShowCardActions] = React.useState(false);

  const Card = Resources.resolved["Card.js"];
  const CardTitle = Resources.resolved["CardTitle.js"];
  const { formatTimestamp } = Resources.resolved["utils.js"];

  const { useSnackbar } = Resources.resolved["hooks.js"];
  const { showSnackbar } = useSnackbar();

  const zipReference = item.keywords?.zip_reference?.[0];
  const wfiStatus = item.keywords?.wfi_status?.[0];
  const isSelected = itemsSelectedBulkAssign.selectedIds.has(item.id);

  // From original squirro-integration repo
  const cardProps = {
    cardModes: props.cardModes,
    className: props.className,
    hasEntities:
      props.entities?.length > 0 && props.openLinks === "showDetailTwoColumn",
    hasSubpanel: props.openLinks === "showDetailThreeColumn",
    hasThumbnail: props.showThumbnail,
    height: "auto",
    isActive: props.isActiveItem,
    isFeedbackEnabled: props.isFeedbackEnabled,
    isSidePanelOpen: props.isSidePanelOpen,
    onClick: props.onCardClick,
    showSubPanel: props.showSubPanel,
    width: props.width,
  };

  // From original squirro-integration repo
  const cardActionsProps = {
    hideReferences: false,
    isFeedbackEnabled: props.isFeedbackEnabled,
    isStarred: props.isStarred,
    itemId: props.itemId,
    itemRank: props.itemRank,
    logFeedback: props.logFeedback,
    numberOfNotes: props.showNotes && props.notes?.length,
    onReferencesClick: props.onReferencesClick,
    onStarClick: props.onStarClick,
    onTranslateClick: props.onTranslateClick,
    pdfUrl: props.pdfUrl,
    references: props.references,
  };

  const updateItemsSelectedBulkAssign = () => {
    const maxItemsSelectedAllowed = 30;
    const { selectedIds: updatedIds } = itemsSelectedBulkAssign;

    if (!isSelected) {
      if (updatedIds.size === 0) {
        // First document selected, track the status
        updatedIds.add(item.id);
        setItemsSelectedBulkAssign({
          currentDocStatus: {
            code: item.keywords?.current_doc_status?.[0],
            header: item.keywords?.current_doc_status_header?.[0],
          },
          selectedIds: new Set(updatedIds),
        });
        return;
      }

      // Limit the number of documents that can be selected
      if (updatedIds.size >= maxItemsSelectedAllowed) {
        return showSnackbar(
          `Maximum ${maxItemsSelectedAllowed} documents can be selected at one time.`,
          "error",
        );
      }

      // Subsequent documents selected. Check if they have the same status
      if (
        item.keywords?.current_doc_status?.[0] !==
        itemsSelectedBulkAssign.currentDocStatus.code
      ) {
        return showSnackbar(
          "Please select documents of the same status.",
          "error",
        );
      }

      // Selected documents now have the same status
      updatedIds.add(item.id);
      setItemsSelectedBulkAssign({
        ...itemsSelectedBulkAssign,
        selectedIds: new Set(updatedIds),
      });
    } else {
      updatedIds.delete(item.id);

      if (updatedIds.size === 0) {
        // No document selected anymore. Reset the state
        setItemsSelectedBulkAssign({
          currentDocStatus: {
            code: "",
            header: "",
          },
          selectedIds: new Set(),
        });
      } else {
        // Deselect the document
        setItemsSelectedBulkAssign({
          ...itemsSelectedBulkAssign,
          selectedIds: new Set(updatedIds),
        });
      }
    }
  };

  return (
    <div className="itemsWidget__singleCard">
      <Card
        {...cardProps}
        onMouseEnter={() => setShowCardActions(true)}
        onMouseLeave={() => setShowCardActions(false)}
        onBlur={() => setShowCardActions(false)}
        sx={{
          pr: 2,
          pl: dashboardTitle === "WIP" ? 0 : 2,
          py: 1.5,
          position: "relative",
          alignItems: "center",
          borderLeft: 1,
          borderRight: 1,
          borderColor: "divider",
        }}
      >
        <MUI.Grid container spacing={1.5} sx={{ alignItems: "center" }}>
          {dashboardTitle === "WIP" && (
            <>
              <MUI.Grid item xs={0.5}>
                <MUI.Checkbox
                  sx={{ position: "absolute", top: 0, bottom: 0 }}
                  onClick={(e) => {
                    e.stopPropagation();
                    updateItemsSelectedBulkAssign();
                  }}
                  disableRipple
                  checked={isSelected}
                />
              </MUI.Grid>
            </>
          )}
          <MUI.Grid item xs={1}>
            <ListViewText>
              {props.keywords?.current_doc_status_header?.[0] || "-"}
            </ListViewText>
          </MUI.Grid>
          <MUI.Grid item xs={1}>
            <>
              {zipReference && (
                <MUI.Tooltip
                  title={zipReference.split("|")[0]}
                  placement="top"
                  disableInteractive
                >
                  <MUI.Icon
                    fontSize="small"
                    sx={{
                      color: theme.palette.text.secondary,
                      mr: "2px",
                      verticalAlign: "sub",
                    }}
                  >
                    folder_zip
                  </MUI.Icon>
                </MUI.Tooltip>
              )}
              {wfiStatus === "Failure" && (
                <MUI.Tooltip
                  title="WFI Upload Failed"
                  placement="top"
                  disableInteractive
                >
                  <MUI.Icon
                    fontSize="small"
                    sx={{
                      color: theme.palette.text.secondary,
                      mr: "2px",
                      verticalAlign: "sub",
                    }}
                  >
                    error
                  </MUI.Icon>
                </MUI.Tooltip>
              )}
            </>
            <CardTitle
              className="listview-title"
              color="textPrimary"
              variant="body2"
              titleContent={props.escapedTitle}
            />
          </MUI.Grid>
          <MUI.Grid item xs={dashboardTitle === "WIP" ? 1.3 : 1.8}>
            <ListViewText>
              {props.keywords?.company_name?.join(", ") || "-"}
            </ListViewText>
          </MUI.Grid>
          <MUI.Grid item xs={1}>
            <ListViewText>{props.keywords?.cso_name?.[0] || "-"}</ListViewText>
          </MUI.Grid>
          <MUI.Grid item xs={1.1}>
            <ListViewText>
              {props.keywords?.document_type?.[0] || "-"}
            </ListViewText>
          </MUI.Grid>

          <MUI.Grid item xs={1}>
            <ListViewText>
              {props.keywords?.document_date
                ? moment(props.keywords?.document_date[0], "YYYY-MM-DD").format(
                    "DD/MM/YYYY",
                  )
                : "-"}
            </ListViewText>
          </MUI.Grid>

          <MUI.Grid item xs={1}>
            <ListViewText>
              {props.keywords?.references?.join(", ") || "-"}
            </ListViewText>
          </MUI.Grid>
          <MUI.Grid item xs={1}>
            <ListViewText>{props.keywords?.rm_name?.[0] || "-"}</ListViewText>
          </MUI.Grid>
          <MUI.Grid item xs={1.1}>
            <ListViewText>
              {item.created_at ? formatTimestamp(item.created_at) : "-"}
            </ListViewText>
          </MUI.Grid>
          <MUI.Grid item xs={1}>
            <ListViewText>{item.keywords?.user_name?.[0] || "-"}</ListViewText>
          </MUI.Grid>
          <MUI.Grid item xs={1}>
            <ListViewText>{item.keywords?.num_pages?.[0] || "-"}</ListViewText>
          </MUI.Grid>
        </MUI.Grid>
        <MUI.Fade
          in={showCardActions}
          timeout={200}
          easing={{ enter: "ease-in-out" }}
          exit={false}
        >
          <MUI.Box
            sx={(theme) => ({
              display: "flex",
              flexDirection: "column",
              justifyContent: "center",
              position: "absolute",
              backgroundColor: "inherit",
              boxShadow: `-24px 0px 18px 0px ${theme.palette.hover}`,
              pl: 1,
              height: "100%",
              right: 12,
            })}
          >
            <props.overrides.CardActions {...cardActionsProps} />
          </MUI.Box>
        </MUI.Fade>
      </Card>
    </div>
  );
};

const ListViewText = ({ children }) => {
  return (
    <MUI.Typography
      sx={{
        textOverflow: "ellipsis",
        overflow: "hidden",
        wordWrap: "break-word",
        wordBreak: "break-word",
        display: "-webkit-box",
        overflow: "hidden",
        WebkitBoxOrient: "vertical",
        WebkitLineClamp: "5",
      }}
      color="textPrimary"
      variant="body2"
      component="p"
    >
      {children}
    </MUI.Typography>
  );
};
