const AutocompleteFromAPI = ({
  label,
  multiple,
  value,
  onChange,
  autoCompleteAPIUrl,
  fetchOptionsOnInputChange,
  getOptions,
  ...props
}) => {
  const [open, setOpen] = React.useState(false);
  const [loading, setLoading] = React.useState(false);
  const [options, setOptions] = React.useState([]);

  React.useEffect(() => {
    if (!open && fetchOptionsOnInputChange) {
      setOptions([]);
    } else if (open && !options.length) {
      setLoading(true);
    }
  }, [open]);

  React.useEffect(() => {
    if (!loading) return undefined;

    fetch(autoCompleteAPIUrl)
      .then((res) => res.json())
      .then((data) => {
        getOptions ? setOptions(getOptions(data)) : setOptions(data);
      })
      .catch(() => setOptions([]))
      .finally(() => setLoading(false));
  }, [loading]);

  let timer;
  const handleInputChange = (event, value) => {
    if (timer) clearTimeout(timer);
    timer = setTimeout(() => {
      fetch(autoCompleteAPIUrl + value)
        .then((res) => res.json())
        .then((data) => setOptions(data))
        .catch(() => setOptions([]));
    }, 300);
  };

  return (
    <MUI.Autocomplete
      open={open}
      multiple={multiple}
      loading={loading}
      autoHighlight
      onOpen={() => setOpen(true)}
      onClose={() => setOpen(false)}
      value={value}
      onChange={onChange}
      options={options}
      {...(fetchOptionsOnInputChange && {
        filterOptions: (x) => x,
        onInputChange: handleInputChange,
      })}
      renderInput={(params) => (
        <MUI.TextField
          {...params}
          label={label}
          InputProps={{
            ...params.InputProps,
            endAdornment: (
              <>
                {loading && <MUI.CircularProgress color="inherit" size={20} />}
                {params.InputProps.endAdornment}
              </>
            ),
          }}
        />
      )}
      {...props}
    />
  );
};

const FreeTextMultiple = ({
  label,
  value,
  setValue,
  setIsFormDisabled,
  valueDelimiter = "",
  shouldValidateInput = true,
}) => {
  const [inputValue, setInputValue] = React.useState("");
  const [errorMsg, setErrorMsg] = React.useState("");
  const valueCombined = [...value, inputValue].join(valueDelimiter);

  React.useEffect(() => {
    shouldValidateInput && validateInput(valueCombined);
  }, [valueCombined]);

  React.useEffect(() => {
    setIsFormDisabled(Boolean(errorMsg));
  }, [errorMsg]);

  const validateInput = (input, maxChars = 250) => {
    if (input.length > maxChars) {
      setErrorMsg("Maximum character limit exceeded");
      return false;
    }

    setErrorMsg("");
    return true;
  };

  return (
    <MUI.Autocomplete
      multiple
      freeSolo
      open={false}
      value={value}
      inputValue={inputValue}
      onChange={(_, newValue) => {
        if (!errorMsg) {
          setValue(newValue);
          setInputValue("");
        }
      }}
      onInputChange={(e) => {
        setInputValue(e.target.value || "");
      }}
      options={[]}
      renderInput={(params) => (
        <MUI.TextField
          {...params}
          error={errorMsg}
          helperText={errorMsg}
          variant="outlined"
          label={label}
          onBlur={() => {
            if (inputValue && !errorMsg) {
              setValue([...value, inputValue]);
              setInputValue("");
            }
          }}
        />
      )}
    />
  );
};

export default { AutocompleteFromAPI, FreeTextMultiple };
