const EditForm = ({ item, handleClose, activeItemModel }) => {
  const [loading, setLoading] = React.useState(false);
  const [isDisabled, setIsDisabled] = React.useState(false);
  const [isDatePickerOpen, setIsDatePickerOpen] = React.useState(false);
  const [hasLabelChanged, setHasLabelChanged] = React.useState({
    company_name: false,
    document_type: false,
    document_format: false,
    document_date: false,
    cso_name: false,
    rm_name: false,
    references: false,
    notes: false,
  });

  const widget = React.useContext(WidgetContexts.Items);
  const itemModel = widget.props?.collection?.get(item.id);
  const { project, user, additionalWidgetConfig, dashboardTitle } =
    widget.hasOwnProperty("mainJsProps") ? widget.mainJsProps : widget;
  const user_role = user?.user_information?.role_ocbc?.[0];

  const { updateDocument } = Resources.resolved["api.js"];
  const { useSnackbar } = Resources.resolved["hooks.js"];
  const { AutocompleteFromAPI, FreeTextMultiple } =
    Resources.resolved["Inputs.js"];

  const oldLabels = {
    companyName: item.keywords.company_name || [],
    documentType: item.keywords.document_type?.[0],
    documentFormat: item.keywords.document_format?.[0],
    documentDate: item.keywords.document_date
      ? moment(item.keywords.document_date[0], "YYYY-MM-DD")
      : null,
    csoName: item.keywords.cso_name?.[0],
    rmName: item.keywords.rm_name?.[0],
    references: item.keywords.references || [],
    notes: item.keywords.notes || [],
  };

  const [documentType, setDocumentType] = React.useState(
    oldLabels.documentType,
  );
  const [documentFormat, setDocumentFormat] = React.useState(
    oldLabels.documentFormat || null,
  );
  const [companyNames, setCompanyNames] = React.useState(oldLabels.companyName);
  const [documentDate, setDocumentDate] = React.useState(
    oldLabels.documentDate,
  );
  const [csoName, setCSOName] = React.useState(oldLabels.csoName);
  const [rmName, setRMName] = React.useState(oldLabels.rmName);
  const [references, setReferences] = React.useState(oldLabels.references);
  const [notes, setNotes] = React.useState(oldLabels.notes);

  const { showSnackbar } = useSnackbar();

  React.useEffect(() => {
    setHasLabelChanged({
      ...hasLabelChanged,
      document_type: documentType !== oldLabels.documentType,
    });
  }, [documentType]);

  React.useEffect(() => {
    setHasLabelChanged({
      ...hasLabelChanged,
      document_format: documentFormat !== oldLabels.documentFormat,
    });
  }, [documentFormat]);

  React.useEffect(() => {
    setHasLabelChanged({
      ...hasLabelChanged,
      cso_name: csoName !== oldLabels.csoName,
    });
  }, [csoName]);

  React.useEffect(() => {
    setHasLabelChanged({
      ...hasLabelChanged,
      rm_name: rmName !== oldLabels.rmName,
    });
  }, [rmName]);

  React.useEffect(() => {
    setHasLabelChanged({
      ...hasLabelChanged,
      references: references !== oldLabels.references,
    });
  }, [references]);

  React.useEffect(() => {
    setHasLabelChanged({
      ...hasLabelChanged,
      notes: notes !== oldLabels.notes,
    });
  }, [notes]);

  React.useEffect(() => {
    if (documentDate) {
      setHasLabelChanged({
        ...hasLabelChanged,
        document_date:
          !oldLabels.documentDate ||
          documentDate.format("YYYY-MM-DD") !==
            oldLabels.documentDate.format("YYYY-MM-DD"),
      });
    }
  }, [documentDate]);

  React.useEffect(() => {
    let hasCompanyNameChanged =
      companyNames.length !== oldLabels.companyName.length;

    if (!hasCompanyNameChanged) {
      for (let i = 0; i < companyNames.length; i++) {
        if (companyNames[i] !== oldLabels.companyName[i]) {
          hasCompanyNameChanged = true;
          break;
        }
      }
    }

    setHasLabelChanged({
      ...hasLabelChanged,
      company_name: hasCompanyNameChanged,
    });
  }, [companyNames]);

  const updateModelKeywords = (updatedKeywords) => {
    if (itemModel) {
      // Update from item cards
      itemModel.set("keywords", {
        ...item.keywords,
        ...updatedKeywords,
      });
      widget.props.collection.trigger("updated:model");
    } else {
      // Update from item detail
      activeItemModel?.set("keywords", {
        ...item.keywords,
        ...updatedKeywords,
      });
      activeItemModel?.trigger("updated:model");
    }
  };

  const handleSubmit = async (e) => {
    const wfiDocumentId = item.keywords.wfi_document_id?.[0];
    const wfiCompanyName = item.keywords.wfi_company_name?.[0];

    if (!wfiDocumentId) {
      showSnackbar(
        "This document is not in WFI yet. Please try again later.",
        "error",
      );
      handleClose(e);
      return;
    }

    let labels = {
      ...(documentType &&
        hasLabelChanged.document_type && {
          document_type: [documentType],
        }),
      ...(documentFormat &&
        hasLabelChanged.document_format && {
          document_format: [documentFormat],
        }),
      ...(companyNames.length &&
        hasLabelChanged.company_name && {
          company_name: companyNames,
        }),
      ...(documentDate &&
        hasLabelChanged.document_date && {
          document_date: [documentDate.format("YYYY-MM-DD")],
        }),
      ...(csoName &&
        hasLabelChanged.cso_name && {
          cso_name: [csoName],
        }),
      ...(rmName &&
        hasLabelChanged.rm_name && {
          rm_name: [rmName],
        }),
      ...(hasLabelChanged.references && {
        references: references,
      }),
      ...(hasLabelChanged.notes && {
        notes: notes,
      }),
      ...(hasLabelChanged.company_name &&
        (!wfiCompanyName || !companyNames.includes(wfiCompanyName)) && {
          wfi_company_name: [companyNames[0]],
        }),
    };

    const activityTrackingLabels = {
      ...(documentType &&
        hasLabelChanged.document_type && {
          document_type: {
            "document_type.old": [oldLabels.documentType],
            "document_type.current": [documentType],
          },
        }),
      ...(companyNames.length > 0 &&
        hasLabelChanged.company_name && {
          company_name: {
            "company_name.old": oldLabels.companyName,
            "company_name.current": companyNames,
          },
        }),
      ...(documentDate &&
        hasLabelChanged.document_date && {
          document_date: {
            "document_date.old": [
              oldLabels.documentDate &&
                oldLabels.documentDate.format("YYYY-MM-DD"),
            ],
            "document_date.current": [documentDate.format("YYYY-MM-DD")],
          },
        }),
    };

    setLoading(true);

    try {
      const response = await updateDocument(
        item.id,
        wfiDocumentId,
        project.id,
        labels,
      );

      const data = await response.json();
      labels = { ...labels, ...data };
      updateModelKeywords(labels);
      for (const label in activityTrackingLabels) {
        const activityPayload = {
          action: "document.manual_classification",
          label: getLabelName(label),
          item: {
            id: item.id,
            title: item.title,
          },
          company_name: companyNames,
          user_name: user.fullName,
          user_email: user.email,
          project: {
            id: project.id,
            title: project.title,
          },
          ...activityTrackingLabels[label],
        };
        logActivity(activityPayload);
      }
      showSnackbar("Document has been updated.", "success");
    } catch {
      showSnackbar("Failed to update labels. Please try again later.", "error");
    }
    setLoading(false);
    handleClose(e);
  };

  return (
    <MUI.Box
      sx={{
        position: "absolute",
        top: "50%",
        left: "50%",
        display: "flex",
        flexDirection: "column",
        maxHeight: "90%",
        transform: "translate(-50%, -50%)",
        margin: "auto",
        backgroundColor: "#ffffff",
        boxShadow: 24,
        borderRadius: 1,
        width: {
          lg: "50%",
          xs: "80%",
        },
      }}
      onClick={(e) => {
        e.stopPropagation();

        // DatePicker does not close when clicked outside. Close the DatePicker if it
        // is clicked outside of the DatePicker and still in the modal.
        const closeDatePicker =
          !$.contains(document.querySelector("#date-picker"), e.target) &&
          !$.contains(
            document.querySelector(".MuiPickersPopper-root"),
            e.target,
          );
        if (closeDatePicker) {
          setIsDatePickerOpen(false);
        }
      }}
    >
      <MUI.Stack spacing={2} sx={{ p: 3, height: "100%", overflowY: "auto" }}>
        <MUI.Typography
          sx={{ mt: "0 !important", fontWeight: "bold" }}
          variant="body1"
          component="h4"
        >
          Edit
        </MUI.Typography>
        {additionalWidgetConfig.editFormFields.map(({ facet }) => {
          switch (facet) {
            case "company_name":
              return (
                <AutocompleteFromAPI
                  label="Company Name(s)"
                  multiple
                  value={companyNames}
                  onChange={(e, newCompanyNames) => {
                    setCompanyNames(
                      newCompanyNames.map(
                        (companyName) => companyName.name ?? companyName,
                      ),
                    );
                  }}
                  autoCompleteAPIUrl={`/studio/autocomplete_api/company_names/${project.id}?q=`}
                  fetchOptionsOnInputChange
                  isOptionEqualToValue={(option, value) =>
                    option.name === value
                  }
                  renderOption={(props, option) => (
                    <li {...props}>
                      <MUI.Box
                        sx={{
                          width: "100%",
                          display: "flex",
                          justifyContent: "space-between",
                          alignItems: "center",
                        }}
                      >
                        <span>{option.name}</span>
                        <span style={{ color: "gray" }}>{option.cif}</span>
                      </MUI.Box>
                    </li>
                  )}
                />
              );
            case "document_format":
              return (
                <MUI.FormControl>
                  <MUI.InputLabel>Document Format</MUI.InputLabel>
                  <MUI.Select
                    label="Document Format"
                    value={documentFormat}
                    onChange={(e) => setDocumentFormat(e.target.value)}
                    disabled={
                      oldLabels.documentFormat || dashboardTitle !== "WIP"
                    }
                  >
                    <MUI.MenuItem value="Hardcopy - Original">
                      Hardcopy - Original
                    </MUI.MenuItem>
                    <MUI.MenuItem value="Hardcopy - Photocopy">
                      Hardcopy - Photocopy
                    </MUI.MenuItem>
                    <MUI.MenuItem value="Softcopy">Softcopy</MUI.MenuItem>
                  </MUI.Select>
                </MUI.FormControl>
              );
            case "document_type":
              return (
                <AutocompleteFromAPI
                  label="Document Type"
                  value={documentType}
                  onChange={(e, docs) => setDocumentType(docs)}
                  autoCompleteAPIUrl={`/studio/autocomplete_api/document_types/${project.id}`}
                />
              );
            case "document_date":
              return (
                <DocumentDatePicker
                  open={isDatePickerOpen}
                  onOpen={() => setIsDatePickerOpen(true)}
                  onClose={() => setIsDatePickerOpen(false)}
                  documentDate={documentDate}
                  setDocumentDate={setDocumentDate}
                />
              );
            case "cso_name":
              return (
                <AutocompleteFromAPI
                  label="CSO Name"
                  value={csoName}
                  onChange={(e, name) => setCSOName(name)}
                  autoCompleteAPIUrl={`/studio/autocomplete_api/cso_names/${project.id}?q=`}
                  fetchOptionsOnInputChange
                  disabled={user_role.startsWith("Workbench:CS")}
                />
              );
            case "rm_name":
              return (
                <AutocompleteFromAPI
                  label="RM Name"
                  value={rmName}
                  onChange={(e, name) => setRMName(name)}
                  autoCompleteAPIUrl={`/studio/autocomplete_api/rm_names/${project.id}?q=`}
                  fetchOptionsOnInputChange
                  disabled={
                    user_role.startsWith("Workbench:CS") ||
                    user_role.startsWith("iLMS:BBCA") ||
                    user_role.startsWith("iLMS:BLT")
                  }
                />
              );
            case "references":
              return (
                <FreeTextMultiple
                  label="References"
                  value={references}
                  setValue={setReferences}
                  setIsFormDisabled={setIsDisabled}
                  valueDelimiter=";"
                />
              );
            case "notes":
              return (
                <FreeTextMultiple
                  label="Notes"
                  value={notes}
                  setValue={setNotes}
                  setIsFormDisabled={setIsDisabled}
                />
              );
          }
        })}

        <MUI.Stack
          direction="row"
          justifyContent="flex-end"
          spacing={2}
          sx={{ pt: 3 }}
        >
          <MUI.Button onClick={handleClose}>Cancel</MUI.Button>
          <MUI.Button
            variant="contained"
            disabled={isDisabled}
            onClick={handleSubmit}
          >
            Save
          </MUI.Button>
        </MUI.Stack>
      </MUI.Stack>

      <MUI.Backdrop
        sx={{
          backgroundColor: "rgba(255, 255, 255, 0.5)",
        }}
        open={loading}
      >
        <MUI.CircularProgress />
      </MUI.Backdrop>
    </MUI.Box>
  );
};

const DocumentDatePicker = ({
  open,
  onOpen,
  onClose,
  documentDate,
  setDocumentDate,
}) => {
  return (
    <div id="date-picker">
      <MUIXDatePicker.LocalizationProvider dateAdapter={MUIXAdapterMoment}>
        <MUIXDatePicker.DatePicker
          open={open}
          onOpen={onOpen}
          label="Document Date"
          value={documentDate}
          onChange={(date) => {
            setDocumentDate(date);
          }}
          onClose={onClose}
          disableFuture={true}
          format="DD/MM/YYYY"
          sx={{
            width: "100%",
          }}
        />
      </MUIXDatePicker.LocalizationProvider>
    </div>
  );
};

const logActivity = async (payload) => {
  await fetch(`/activity`, {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
    },
    body: JSON.stringify(payload),
  });
};

const getLabelName = (label) => {
  // document_type -> Document Type
  return label.replace(/(^|_)\S/g, (match) =>
    match.toUpperCase().replace("_", " "),
  );
};

export default EditForm;
