const DeleteOrRestoreDocument = ({ disabled, disabledTooltip }) => {
  const [isModalOpen, setIsModalOpen] = React.useState(false);
  const item = React.useContext(ModelContexts.Items);
  const isDeleted = item.keywords?.is_deleted?.[0] === "true";

  const handleOpen = (e) => {
    e.stopPropagation();
    setIsModalOpen(true);
  };

  const handleClose = (e) => {
    e.stopPropagation();
    setIsModalOpen(false);
  };

  return (
    <>
      <MUI.Tooltip
        title={!disabled ? (isDeleted ? "Restore" : "Delete") : disabledTooltip}
        placement="top"
        arrow
        TransitionComponent={MUI.Zoom}
      >
        <span>
          <MUI.IconButton disabled={disabled} size="small" onClick={handleOpen}>
            <MUI.Icon color={disabled ? "disabled" : ""}>
              {isDeleted ? "restore_from_trash" : "delete"}
            </MUI.Icon>
          </MUI.IconButton>
        </span>
      </MUI.Tooltip>
      <MUI.Modal open={isModalOpen} onClose={handleClose}>
        <div>
          <DeleteOrRestoreForm
            isDeleted={isDeleted}
            handleClose={handleClose}
          />
        </div>
      </MUI.Modal>
    </>
  );
};

const DeleteOrRestoreForm = ({ handleClose, isDeleted }) => {
  const widget = React.useContext(WidgetContexts.Items);
  const { customProps } = widget;
  const item = React.useContext(ModelContexts.Items);

  const { useSnackbar } = Resources.resolved["hooks.js"];
  const { updateSquirroLabels } = Resources.resolved["api.js"];
  const { showSnackbar } = useSnackbar();

  const { project, user } = widget.mainJsProps;
  const action = isDeleted ? "restore" : "delete";

  const deleteOrRestoreDocument = async () => {
    try {
      await updateSquirroLabels(item.id, project.id, {
        is_deleted: action === "delete" ? ["true"] : ["false"],
        wfi_document_id: item.keywords?.wfi_document_id,
      });

      const activityPayload = {
        action: `document.${action}`,
        action_type: action.charAt(0).toUpperCase() + action.slice(1),
        item: {
          id: item.id,
          title: item.title,
        },
        user_name: user.fullName,
        user_email: user.email,
        project: {
          id: project.id,
          title: project.title,
        },
      };
      logActivity(activityPayload);
      showSnackbar(`Document has been ${action}d.`, "success");
    } catch (err) {
      showSnackbar(
        `Failed to ${action} document. Please try again later.`,
        "error",
      );
    }
  };

  const handleOkClick = (e) => {
    handleClose(e);
    if (widget.customProps?.isDeleted) {
      const deletedItems = widget.customProps.isDeleted.add(item.id);
      widget.setCustomProps({ ...customProps, isDeleted: deletedItems });
    } else {
      widget.setCustomProps({ ...customProps, isDeleted: new Set([item.id]) });
    }
    deleteOrRestoreDocument();
  };

  const handleCancelClick = (e) => {
    handleClose(e);
  };

  return (
    <MUI.Box
      sx={{
        position: "absolute",
        top: "50%",
        left: "50%",
        transform: "translate(-50%, -50%)",
        margin: "auto",
        backgroundColor: "#ffffff",
        boxShadow: 24,
        borderRadius: "8px",
        width: {
          lg: "50%",
          xs: "80%",
        },
      }}
      onClick={(e) => e.stopPropagation()}
    >
      <MUI.Box sx={{ p: "24px" }}>
        <MUI.Typography
          sx={{ mt: "0 !important" }}
          variant="body1"
          component="h4"
        >
          {`Are you sure you want to ${action} ${
            item.title ?? "this document"
          }?`}
        </MUI.Typography>
      </MUI.Box>
      <MUI.Stack
        direction="row"
        justifyContent="flex-end"
        spacing={2}
        sx={{ p: "24px" }}
      >
        <MUI.Button onClick={handleCancelClick}>Cancel</MUI.Button>
        <MUI.Button variant="contained" onClick={handleOkClick}>
          {action}
        </MUI.Button>
      </MUI.Stack>
    </MUI.Box>
  );
};

const logActivity = async (payload) => {
  await fetch(`/activity`, {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
    },
    body: JSON.stringify(payload),
  });
};

export default DeleteOrRestoreDocument;
