export default (props) => {
  const [loading, setLoading] = React.useState(true);
  const [error, setError] = React.useState("");
  const [stats, setStats] = React.useState({});
  const documentTypes = Object.keys(stats);

  const { addFacetValue, clearSelection } = Hooks.useDrilldown(
    props.dashboardState,
    props.widgetModel,
  );

  const infoText = (
    <>
      <p>Excluding {EXCLUDED_DOCUMENT_TYPES.length} document types</p>
      <ul>
        {EXCLUDED_DOCUMENT_TYPES.map((item) => (
          <li>{item}</li>
        ))}
      </ul>
      <p>
        Accuracy is calculated based on the documents uploaded after Oct 9 for
        which the correct document types have been provided
      </p>
    </>
  );

  const handleChange = (e) => {
    const newDocumentType = e.target.value;
    clearSelection();

    if (newDocumentType === "UNTAGGED") {
      addFacetValue(null, "-document_type_true:*");
    } else if (newDocumentType !== "ALL") {
      addFacetValue("document_type_true", e.target.value);
    }
  };

  React.useEffect(() => {
    const fetchMLStats = async () => {
      try {
        const res = await fetch(
          `/studio/doctype_stats/doctype/${props.project.id}`,
        );
        if (!res.ok) throw new Error("ML accuracy is not available.");
        const { results } = await res.json();

        // Calculate correct and total for all valid doc types
        const overallStats = Object.entries(results).reduce(
          (accumulator, current) => {
            const [docType, values] = current;
            if (
              !EXCLUDED_DOCUMENT_TYPES.includes(docType) &&
              docType !== "ALL"
            ) {
              accumulator.correct += values.correct;
              accumulator.total += values.total;
            }
            return accumulator;
          },
          {
            correct: 0,
            total: 0,
          },
        );
        overallStats.accuracy = Math.floor(
          (overallStats.correct / overallStats.total) * 100,
        );
        results.ALL = overallStats;
        setStats(results);
      } catch (err) {
        setError(err);
      }
      setLoading(false);
    };
    fetchMLStats();
  }, []);

  return (
    <>
      {loading && <MUI.LinearProgress />}
      {error && (
        <MUI.Box>
          <MUI.Alert severity="error">ML accuracy is not available.</MUI.Alert>
        </MUI.Box>
      )}
      {!loading && !error && (
        <MUI.Stack spacing={2} direction="column" sx={{ pb: 2 }}>
          <MUI.Box
            sx={{
              display: "flex",
              alignItems: "center",
              px: 2,
              columnGap: 1,
            }}
          >
            <MUI.Typography variant="h3">ML Statistics</MUI.Typography>
            <MUI.Tooltip title={infoText} disableInteractive>
              <MUI.Icon fontSize="small">info_outlined_icon</MUI.Icon>
            </MUI.Tooltip>
          </MUI.Box>
          <MUI.FormControl fullWidth>
            <MUI.InputLabel id="document-type-selector">
              Document Type
            </MUI.InputLabel>
            <MUI.Select
              labelId="document-type-selector"
              defaultValue="ALL"
              label="Document Type"
              onChange={handleChange}
              MenuProps={{ PaperProps: { sx: { maxHeight: "50%" } } }}
            >
              {documentTypes.map((documentType) => (
                <MUI.MenuItem key={documentType} value={documentType}>
                  {documentType}
                </MUI.MenuItem>
              ))}
              <MUI.MenuItem value="UNTAGGED">UNTAGGED</MUI.MenuItem>
            </MUI.Select>
          </MUI.FormControl>
          <MUI.TableContainer component={MUI.Paper}>
            <MUI.Table stickyHeader>
              <MUI.TableHead>
                <MUI.TableRow>
                  <MUI.TableCell>Document Type</MUI.TableCell>
                  <MUI.TableCell align="right">Accuracy (%)</MUI.TableCell>
                  <MUI.TableCell align="right">Correct</MUI.TableCell>
                  <MUI.TableCell align="right">Total</MUI.TableCell>
                </MUI.TableRow>
              </MUI.TableHead>
              <MUI.TableBody>
                {Object.entries(stats).map(([docType, values]) => (
                  <MUI.TableRow index={docType}>
                    <MUI.TableCell>{docType}</MUI.TableCell>
                    <MUI.TableCell align="right">
                      {values.accuracy}
                    </MUI.TableCell>
                    <MUI.TableCell align="right">
                      {values.correct}
                    </MUI.TableCell>
                    <MUI.TableCell align="right">{values.total}</MUI.TableCell>
                  </MUI.TableRow>
                ))}
              </MUI.TableBody>
            </MUI.Table>
          </MUI.TableContainer>
        </MUI.Stack>
      )}
    </>
  );
};

const EXCLUDED_DOCUMENT_TYPES = [
  "CP",
  "CR",
  "CM",
  "CONSTRUCTION REPORT",
  "DEED OF COVENANTS",
  "DIRECTORS/MANAGERS CERTIFICATE",
  "EXTERNAL CORRESPONDENCES",
  "LETTER",
  "INTERNAL DOCUMENTS (BBCA)",
  "RISK PARTICIPATION AGREEMENT",
  "TERMSHEET",
  "TRANSCRIPT OF VESSEL",
];
