import sys
from configparser import ConfigParser
from io import BytesIO
from types import ModuleType
from unittest.mock import MagicMock

import pytest
import requests
from cryptography.fernet import Fernet
from redis import Redis

import octopus.data.company_data.company_data_index
import octopus.utils
import squirro.common.config
from octopus.clients.wfi_client import init_wfi_client
from squirro_client import SquirroClient

cfg = ConfigParser()
cfg.read_dict(
    {
        "storage_bucket": {
            "directory": "/var/lib/squirro/storage",
        },
        "storage_ocr": {
            "directory": "tests/octopus/pdf/test-data",
        },
        "squirro": {
            "cluster": "https://example.com",
            "environment": "INTERNAL",
            "project_id": "project_id",
            "token": "token",
            "url": "https://ui.com",
        },
        "wfi": {
            "api_key": "test-api-key",
            "channel": "test-channel",
            "domain": "test-domain",
            "objectstore": "test-objectstore",
            "port": "1234",
            "url": "https://test-url@example.com",
            "wfi_api_header_name": "test-api-header-name",
        },
        "redis": {"host": "localhost", "port": "666", "password": "password"},
        "sentinel": {"hosts": "host1,host2", "port": "666", "password": "password"},
        "index": {
            "es_hosts": "localhost:9200",
            "es_auth_user": "username",
            "es_auth_password": "password",
            "es_pem_path": "/path/to/certs.pem",
        },
    }
)


def company_data():
    return [
        {
            "company_cif": ["11111"],
            "company_name": ["Company B"],
            "company_segment_full": ["REAL ESTATE"],
            "company_uid": ["SG_11111"],
            "country": ["SG"],
            "permission_code": ["A1", "B1"],
            "rm_name": ["Company B RM"],
            "wfi_company_rm_code": ["R012"],
            "wfi_company_segment": ["R"],
            "wfi_company_team_code": ["A1"],
            "wfi_company_team_name": ["Team B"],
        },
        {
            "company_cif": ["22222"],
            "company_name": ["Company A"],
            "company_segment_full": ["REAL ESTATE"],
            "company_uid": ["SG_22222"],
            "country": ["SG"],
            "permission_code": ["B1", "C1"],
            "rm_name": ["Company A RM"],
            "wfi_company_rm_code": ["R011"],
            "wfi_company_segment": ["R"],
            "wfi_company_team_code": ["B1"],
            "wfi_company_team_name": ["Team A"],
        },
        {
            "company_cif": ["33333"],
            "company_name": ["Company C"],
            "company_segment_full": ["REAL ESTATE"],
            "company_uid": ["SG_33333"],
            "country": ["SG"],
            "permission_code": ["C1", "D1"],
            "rm_name": ["Company C RM"],
            "wfi_company_rm_code": ["R013"],
            "wfi_company_segment": ["R"],
            "wfi_company_team_code": ["C1"],
            "wfi_company_team_name": ["Team C"],
        },
        {
            "company_cif": ["44444"],
            "company_name": ["Company D"],
            "company_segment_full": ["REAL ESTATE"],
            "company_uid": ["SG_44444"],
            "country": ["SG"],
            "permission_code": ["D1", "E1"],
            "rm_name": ["Company D RM"],
            "wfi_company_rm_code": ["R014"],
            "wfi_company_segment": ["R"],
            "wfi_company_team_code": ["D1"],
            "wfi_company_team_name": ["Team D"],
        },
        {
            "company_cif": ["55555"],
            "company_name": ["Company E"],
            "company_segment_full": ["REAL ESTATE"],
            "company_uid": ["SG_55555"],
            "country": ["SG"],
            "permission_code": ["E1", "F1"],
            "rm_name": ["Company E RM"],
            "wfi_company_rm_code": ["R015"],
            "wfi_company_segment": ["R"],
            "wfi_company_team_code": ["E1"],
            "wfi_company_team_name": ["Team E"],
        },
        {
            "company_cif": ["66666"],
            "company_name": ["Squirro I"],
            "company_segment_full": ["REAL ESTATE"],
            "company_uid": ["SG_66666"],
            "country": ["SG"],
            "permission_code": ["E1", "F1"],
            "rm_name": ["Squirro I RM"],
            "wfi_company_rm_code": ["R016"],
            "wfi_company_segment": ["R"],
            "wfi_company_team_code": ["F1"],
            "wfi_company_team_name": ["Team E"],
        },
        {
            "company_cif": ["77777"],
            "company_name": ["Squirro II"],
            "company_segment_full": ["REAL ESTATE"],
            "company_uid": ["SG_77777"],
            "country": ["SG"],
            "permission_code": ["E1", "F1"],
            "rm_name": ["Squirro II RM"],
            "wfi_company_rm_code": ["R017"],
            "wfi_company_segment": ["R"],
            "wfi_company_team_code": ["G1"],
            "wfi_company_team_name": ["Team E"],
        },
        {
            "company_cif": ["no company name"],
            "company_segment_full": ["REAL ESTATE"],
            "company_uid": ["SG_98765"],
            "country": ["SG"],
            "permission_code": ["E1", "F1"],
            "rm_name": ["no company name RM"],
            "wfi_company_rm_code": ["R015"],
            "wfi_company_segment": ["R"],
            "wfi_company_team_code": ["F1"],
            "wfi_company_team_name": ["Team E"],
        },
    ]


redis_client_mock = MagicMock(spec=Redis)
squirro_client_mock = MagicMock(spec=SquirroClient)


def mock_sys_octopus():
    # octopus.clients
    mock_clients = ModuleType("octopus.clients")
    mock_clients.init_redis_client = lambda: redis_client_mock
    mock_clients.init_squirro_client = lambda _=None: (
        squirro_client_mock,
        cfg["squirro"]["project_id"],
    )
    mock_clients.init_wfi_client = lambda: init_wfi_client(cfg)
    sys.modules["octopus.clients"] = mock_clients

    # octopus.data.company_data_index.CompanyDataIndex
    company_data_index = (
        octopus.data.company_data.company_data_index.CompanyDataIndex.build_index(
            company_data()
        )
    )
    octopus.data.company_data.company_data_index.CompanyDataIndex.build_index = (
        lambda _=None, __=None: company_data_index
    )
    octopus.data.company_data.company_data_index.CompanyDataIndex.load_index = (
        lambda _=None, __=None: company_data_index
    )
    octopus.data.company_data.company_data_index.CompanyDataIndex.save_index = (
        MagicMock()
    )
    sys.modules["octopus.data.company_data.company_data_index"] = (
        octopus.data.company_data.company_data_index
    )

    # octopus.utils
    octopus.utils.compute_hash = lambda _=None: "hash"
    octopus.utils.load_config = lambda _=None: cfg
    sys.modules["octopus.utils"] = octopus.utils


def mock_sys_squirro_config() -> None:
    squirro.common.config.get_config = lambda _: cfg
    sys.modules["squirro.common.config"] = squirro.common.config


def mock_sys_squirro_integration() -> None:
    sys.modules["squirro.integration"] = MagicMock()


def mock_sys_squirro_integration_frontend() -> None:
    sys.modules["squirro.integration.frontend"] = MagicMock()
    sys.modules["squirro.integration.frontend.context"] = MagicMock()


def mock_sys_squirro_storage_handler() -> None:
    class MockStorageHandler:
        instance: "MockStorageHandler"

        def __init__(self, _=None) -> None:
            """Initialize the mock storage handler."""
            self.__class__.instance = self

        open = MagicMock(return_value=BytesIO(b"This is a test"))
        delete = MagicMock()
        _parse_url = MagicMock()

    mock_storage_handler = ModuleType("squirro.lib.storage.handler")
    mock_storage_handler.StorageHandler = MockStorageHandler
    sys.modules["squirro.lib.storage.handler"] = mock_storage_handler


def mock_sys_squirro_pdfconversion_controllers() -> None:
    sys.modules["squirro.service.pdfconversion.controllers"] = MagicMock()


def pytest_configure():
    mock_sys_octopus()
    mock_sys_squirro_config()
    mock_sys_squirro_integration()
    mock_sys_squirro_integration_frontend()
    mock_sys_squirro_storage_handler()
    mock_sys_squirro_pdfconversion_controllers()


@pytest.fixture(autouse=True)
def mock_load_key(monkeypatch) -> MagicMock:
    mock = MagicMock(return_value=Fernet.generate_key().decode())
    monkeypatch.setattr("vault.load_key", mock)
    return mock


@pytest.fixture(autouse=True)
def mock_load_vault(monkeypatch) -> MagicMock:
    mock = MagicMock(return_value={"section": {"field": "value"}})
    monkeypatch.setattr("vault.load_vault", mock)
    return mock


@pytest.fixture(autouse=True)
def mock_pipelet_init(monkeypatch):
    mock = MagicMock(return_value=None)
    monkeypatch.setattr("squirro.sdk.PipeletV1.__init__", mock)
    return mock


@pytest.fixture(autouse=True)
def mock_redis_client() -> MagicMock:
    redis_client_mock.reset_mock(return_value=True, side_effect=True)
    return redis_client_mock


@pytest.fixture(autouse=True)
def mock_requests_get(monkeypatch) -> MagicMock:
    mock = MagicMock(spec=requests.get)
    monkeypatch.setattr("requests.get", mock)
    return mock


@pytest.fixture(autouse=True)
def mock_requests_patch(monkeypatch) -> MagicMock:
    mock = MagicMock(spec=requests.patch)
    monkeypatch.setattr("requests.patch", mock)
    return mock


@pytest.fixture(autouse=True)
def mock_requests_post(monkeypatch) -> MagicMock:
    mock = MagicMock(spec=requests.post)
    monkeypatch.setattr("requests.post", mock)
    return mock


@pytest.fixture(autouse=True)
def mock_requests_put(monkeypatch) -> MagicMock:
    mock = MagicMock(spec=requests.put)
    monkeypatch.setattr("requests.put", mock)
    return mock


@pytest.fixture(autouse=True)
def mock_requests_session(monkeypatch):
    mock = MagicMock(spec=requests.Session)
    monkeypatch.setattr("requests.Session", mock)
    return mock


@pytest.fixture(autouse=True)
def mock_squirro_client() -> MagicMock:
    squirro_client_mock.reset_mock(return_value=True, side_effect=True)
    return squirro_client_mock
